<?php
/**
 * AI Visibility Tracker - Installation Wizard
 * WordPress-like auto-installer
 */

define('INSTALLER_VERSION', '1.0.0');

define('INSTALLER_MODE', true);

session_start();

$step = $_GET['step'] ?? 1;
$errors = [];
$success = false;

if (file_exists(dirname(__DIR__) . '/config.php') && !defined('INSTALLER_MODE')) {
    die('Application is already installed. If you want to reinstall, please remove config.php first.');
}

function checkRequirements()
{
    $baseDir = dirname(__DIR__);
    $requirements = [];
    
    // PHP Version
    $phpVersion = PHP_VERSION;
    $requirements['php_version'] = [
        'name' => 'PHP Version',
        'required' => '8.0.0',
        'current' => $phpVersion,
        'passed' => version_compare($phpVersion, '8.0.0', '>='),
        'message' => version_compare($phpVersion, '8.0.0', '>=') ? 
            'PHP ' . $phpVersion . ' meets requirement' : 
            'PHP ' . $phpVersion . ' is too old. Requires PHP 8.0.0 or higher'
    ];
    
    // PDO MySQL Extension
    $pdoAvailable = extension_loaded('pdo_mysql');
    $requirements['pdo_mysql'] = [
        'name' => 'PDO MySQL Extension',
        'required' => true,
        'current' => $pdoAvailable,
        'passed' => $pdoAvailable,
        'message' => $pdoAvailable ? 
            'PDO MySQL extension is available' : 
            'PDO MySQL extension is not installed. Run: apt-get install php8.0-mysql'
    ];
    
    // JSON Extension
    $jsonAvailable = extension_loaded('json');
    $requirements['json'] = [
        'name' => 'JSON Extension',
        'required' => true,
        'current' => $jsonAvailable,
        'passed' => $jsonAvailable,
        'message' => $jsonAvailable ? 
            'JSON extension is available' : 
            'JSON extension is not installed. This is usually included in PHP core'
    ];
    
    // cURL Extension
    $curlAvailable = extension_loaded('curl');
    $requirements['curl'] = [
        'name' => 'cURL Extension',
        'required' => true,
        'current' => $curlAvailable,
        'passed' => $curlAvailable,
        'message' => $curlAvailable ? 
            'cURL extension is available' : 
            'cURL extension is not installed. Run: apt-get install php8.0-curl'
    ];
    
    // Config Writable
    $configPath = $baseDir . '/config.php';
    $configExists = file_exists($configPath);
    $configWritable = is_writable($configPath) || !$configExists;
    
    if (!$configExists) {
        $configMessage = 'config.php does not exist - will be created during installation';
    } elseif ($configWritable) {
        $configMessage = 'config.php is writable';
    } else {
        $perms = fileperms($configPath);
        $configMessage = 'config.php is not writable. Current permissions: ' . substr(sprintf('%o', $perms), -4) . 
            '. Run: chmod 644 ' . $configPath;
    }
    
    $requirements['config_writable'] = [
        'name' => 'Config File Writable',
        'required' => true,
        'current' => $configWritable,
        'passed' => $configWritable,
        'message' => $configMessage
    ];
    
    // Logs Directory Writable
    $logsPath = $baseDir . '/logs/';
    $logsExists = is_dir($logsPath);
    $logsWritable = $logsExists && is_writable($logsPath);
    
    if (!$logsExists) {
        $logsMessage = 'logs/ directory does not exist. Creating it...';
        try {
            mkdir($logsPath, 0755, true);
            $logsWritable = is_writable($logsPath);
            $logsMessage = $logsWritable ? 
                'logs/ directory created and is writable' : 
                'logs/ directory created but is not writable. Run: chmod 755 ' . $logsPath;
        } catch (Exception $e) {
            $logsMessage = 'Failed to create logs/ directory: ' . $e->getMessage() . 
                '. Run: mkdir -p ' . $logsPath . ' && chmod 755 ' . $logsPath;
        }
    } elseif ($logsWritable) {
        $perms = fileperms($logsPath);
        $logsMessage = 'logs/ directory is writable';
    } else {
        $perms = fileperms($logsPath);
        $logsMessage = 'logs/ directory is not writable. Current permissions: ' . substr(sprintf('%o', $perms), -4) . 
            '. Run: chmod 755 ' . $logsPath . ' or chown www-data:www-data ' . $logsPath;
    }
    
    $requirements['logs_writable'] = [
        'name' => 'Logs Directory Writable',
        'required' => true,
        'current' => $logsWritable,
        'passed' => $logsWritable,
        'path' => $logsPath,
        'message' => $logsMessage
    ];
    
    // Uploads Directory Writable
    $uploadsPath = $baseDir . '/public/uploads/';
    $uploadsExists = is_dir($uploadsPath);
    $uploadsWritable = $uploadsExists && is_writable($uploadsPath);
    
    if (!$uploadsExists) {
        $uploadsMessage = 'public/uploads/ directory does not exist. Creating it...';
        try {
            mkdir($uploadsPath, 0755, true);
            $uploadsWritable = is_writable($uploadsPath);
            $uploadsMessage = $uploadsWritable ? 
                'public/uploads/ directory created and is writable' : 
                'public/uploads/ directory created but is not writable. Run: chmod 755 ' . $uploadsPath;
        } catch (Exception $e) {
            $uploadsMessage = 'Failed to create public/uploads/ directory: ' . $e->getMessage() . 
                '. Run: mkdir -p ' . $uploadsPath . ' && chmod 755 ' . $uploadsPath;
        }
    } elseif ($uploadsWritable) {
        $uploadsMessage = 'public/uploads/ directory is writable';
    } else {
        $perms = fileperms($uploadsPath);
        $uploadsMessage = 'public/uploads/ directory is not writable. Current permissions: ' . substr(sprintf('%o', $perms), -4) . 
            '. Run: chmod 755 ' . $uploadsPath . ' or chown www-data:www-data ' . $uploadsPath;
    }
    
    $requirements['uploads_writable'] = [
        'name' => 'Uploads Directory Writable',
        'required' => true,
        'current' => $uploadsWritable,
        'passed' => $uploadsWritable,
        'path' => $uploadsPath,
        'message' => $uploadsMessage
    ];
    
    return $requirements;
}

function testDatabaseConnection($host, $database, $username, $password)
{
    try {
        $dsn = "mysql:host={$host};charset=utf8mb4";
        $pdo = new PDO($dsn, $username, $password, [
            PDO::ATTR_ERRMODE => PDO::ERRMODE_EXCEPTION,
            PDO::ATTR_TIMEOUT => 5
        ]);
        
        $pdo->query("USE `{$database}`");
        
        return ['success' => true, 'message' => 'Connection successful'];
    } catch (PDOException $e) {
        if (strpos($e->getMessage(), 'Unknown database') !== false) {
            try {
                $dsn = "mysql:host={$host};charset=utf8mb4";
                $pdo = new PDO($dsn, $username, $password, [
                    PDO::ATTR_ERRMODE => PDO::ERRMODE_EXCEPTION
                ]);
                $pdo->query("CREATE DATABASE `{$database}` CHARACTER SET utf8mb4 COLLATE utf8mb4_unicode_ci");
                return ['success' => true, 'message' => 'Database created successfully'];
            } catch (PDOException $e2) {
                return ['success' => false, 'message' => 'Failed to create database: ' . $e2->getMessage()];
            }
        }
        return ['success' => false, 'message' => 'Connection failed: ' . $e->getMessage()];
    }
}

function createConfigFile($config)
{
    $configContent = "<?php
/**
 * AI Visibility Tracker Configuration
 * Auto-generated by installer
 */

// Database Configuration
define('DB_HOST', '{$config['db_host']}');
define('DB_NAME', '{$config['db_name']}');
define('DB_USER', '{$config['db_user']}');
define('DB_PASS', '{$config['db_pass']}');

// Application Configuration
define('BASE_URL', '{$config['base_url']}');
define('SITE_NAME', '{$config['site_name']}');

// Security Configuration
define('SECRET_KEY', '{$config['secret_key']}');
define('DEBUG_MODE', false);
define('LOG_LEVEL', 6); // 0=EMERGENCY, 7=DEBUG

// Email Configuration
define('ADMIN_EMAIL', '{$config['admin_email']}');
define('EMAIL_FROM', 'noreply@' . \$_SERVER['HTTP_HOST']);

// Rate Limiting
define('MAX_REQUESTS_PER_MINUTE', 60);
define('MAX_REQUESTS_PER_HOUR', 1000);

// Queue Configuration
define('QUEUE_WORKERS', 1);
define('JOB_TIMEOUT', 300);

// OpenRouter API Configuration
define('OPENROUTER_API_KEY', '{$config['openrouter_api_key']}');

// Session Configuration
ini_set('session.cookie_httponly', 1);
ini_set('session.use_strict_mode', 1);
ini_set('session.cookie_samesite', 'Strict');

// Timezone
date_default_timezone_set('UTC');

// Load core classes
require_once dirname(__DIR__) . '/core/Logger.php';
require_once dirname(__DIR__) . '/core/Database.php';
require_once dirname(__DIR__) . '/core/Auth.php';
require_once dirname(__DIR__) . '/core/Queue.php';
";

    $configPath = dirname(__DIR__) . '/config.php';
    
    if (file_put_contents($configPath, $configContent) === false) {
        return false;
    }
    
    chmod($configPath, 0644);
    
    return true;
}

function installDatabase($host, $database, $username, $password)
{
    try {
        $dsn = "mysql:host={$host};dbname={$database};charset=utf8mb4";
        $pdo = new PDO($dsn, $username, $password, [
            PDO::ATTR_ERRMODE => PDO::ERRMODE_EXCEPTION
        ]);
        
        $schema = file_get_contents(__DIR__ . '/schema.sql');
        
        $pdo->exec($schema);
        
        return ['success' => true, 'message' => 'Database installed successfully'];
    } catch (PDOException $e) {
        return ['success' => false, 'message' => 'Database installation failed: ' . $e->getMessage()];
    }
}

function createAdminUser($host, $database, $username, $password, $adminEmail, $adminPassword)
{
    try {
        $dsn = "mysql:host={$host};dbname={$database};charset=utf8mb4";
        $pdo = new PDO($dsn, $username, $password, [
            PDO::ATTR_ERRMODE => PDO::ERRMODE_EXCEPTION
        ]);
        
        $passwordHash = password_hash($adminPassword, PASSWORD_BCRYPT, ['cost' => 12]);
        
        $stmt = $pdo->prepare("INSERT INTO users (email, password_hash, role, is_active, created_at) VALUES (?, ?, 'admin', 1, NOW())");
        $stmt->execute([$adminEmail, $passwordHash]);
        
        return ['success' => true, 'message' => 'Admin user created successfully'];
    } catch (PDOException $e) {
        return ['success' => false, 'message' => 'Admin creation failed: ' . $e->getMessage()];
    }
}

if ($_SERVER['REQUEST_METHOD'] === 'POST') {
    switch ($step) {
        case 2:
            $dbHost = $_POST['db_host'] ?? 'localhost';
            $dbName = $_POST['db_name'] ?? '';
            $dbUser = $_POST['db_user'] ?? '';
            $dbPass = $_POST['db_pass'] ?? '';
            
            if (empty($dbHost) || empty($dbName) || empty($dbUser)) {
                $errors[] = 'Please fill in all required database fields.';
            } else {
                $testResult = testDatabaseConnection($dbHost, $dbName, $dbUser, $dbPass);
                if (!$testResult['success']) {
                    $errors[] = $testResult['message'];
                } else {
                    $_SESSION['db_host'] = $dbHost;
                    $_SESSION['db_name'] = $dbName;
                    $_SESSION['db_user'] = $dbUser;
                    $_SESSION['db_pass'] = $dbPass;
                    header('Location: ?step=3');
                    exit;
                }
            }
            break;
            
        case 3:
            $siteName = $_POST['site_name'] ?? '';
            $baseUrl = $_POST['base_url'] ?? '';
            $adminEmail = $_POST['admin_email'] ?? '';
            $adminPassword = $_POST['admin_password'] ?? '';
            $confirmPassword = $_POST['confirm_password'] ?? '';
            $openrouterKey = $_POST['openrouter_api_key'] ?? '';
            
            if (empty($siteName) || empty($baseUrl) || empty($adminEmail) || empty($adminPassword)) {
                $errors[] = 'Please fill in all required fields.';
            } elseif ($adminPassword !== $confirmPassword) {
                $errors[] = 'Passwords do not match.';
            } elseif (!filter_var($adminEmail, FILTER_VALIDATE_EMAIL)) {
                $errors[] = 'Please enter a valid email address.';
            } elseif (strlen($adminPassword) < 8) {
                $errors[] = 'Password must be at least 8 characters long.';
            } else {
                $_SESSION['site_name'] = $siteName;
                $_SESSION['base_url'] = rtrim($baseUrl, '/');
                $_SESSION['admin_email'] = $adminEmail;
                $_SESSION['admin_password'] = $adminPassword;
                $_SESSION['openrouter_api_key'] = $openrouterKey;
                
                $secretKey = bin2hex(random_bytes(32));
                $_SESSION['secret_key'] = $secretKey;
                
                header('Location: ?step=4');
                exit;
            }
            break;
            
        case 4:
            $config = [
                'db_host' => $_SESSION['db_host'],
                'db_name' => $_SESSION['db_name'],
                'db_user' => $_SESSION['db_user'],
                'db_pass' => $_SESSION['db_pass'],
                'base_url' => $_SESSION['base_url'],
                'site_name' => $_SESSION['site_name'],
                'admin_email' => $_SESSION['admin_email'],
                'secret_key' => $_SESSION['secret_key'],
                'openrouter_api_key' => $_SESSION['openrouter_api_key']
            ];
            
            $dbResult = installDatabase($config['db_host'], $config['db_name'], $config['db_user'], $config['db_pass']);
            
            if (!$dbResult['success']) {
                $errors[] = $dbResult['message'];
            } else {
                $adminResult = createAdminUser(
                    $config['db_host'],
                    $config['db_name'],
                    $config['db_user'],
                    $config['db_pass'],
                    $config['admin_email'],
                    $_SESSION['admin_password']
                );
                
                if (!$adminResult['success']) {
                    $errors[] = $adminResult['message'];
                } else {
                    if (createConfigFile($config)) {
                        $success = true;
                        
                        $lockContent = "<?php\n// Installation completed at " . date('Y-m-d H:i:s') . "\n// This file prevents reinstallation\nexit;\n";
                        file_put_contents(__DIR__ . '/install.lock', $lockContent);
                        
                        session_destroy();
                    } else {
                        $errors[] = 'Failed to create configuration file. Please check file permissions.';
                    }
                }
            }
            break;
    }
}

$requirements = checkRequirements();
$allRequirementsMet = true;
foreach ($requirements as $req) {
    if (!$req['passed']) {
        $allRequirementsMet = false;
        break;
    }
}

?>
<!DOCTYPE html>
<html lang="en">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>AI Visibility Tracker - Installation</title>
    <style>
        * {
            margin: 0;
            padding: 0;
            box-sizing: border-box;
        }
        
        body {
            font-family: -apple-system, BlinkMacSystemFont, 'Segoe UI', Roboto, sans-serif;
            background: #f5f7fa;
            color: #333;
            line-height: 1.6;
        }
        
        .container {
            max-width: 800px;
            margin: 0 auto;
            padding: 40px 20px;
        }
        
        .header {
            text-align: center;
            margin-bottom: 40px;
        }
        
        .header h1 {
            font-size: 2.5rem;
            color: #2563eb;
            margin-bottom: 10px;
        }
        
        .header p {
            color: #64748b;
            font-size: 1.1rem;
        }
        
        .progress-bar {
            background: #e2e8f0;
            height: 8px;
            border-radius: 4px;
            margin-bottom: 40px;
            overflow: hidden;
        }
        
        .progress-fill {
            background: linear-gradient(90deg, #3b82f6, #1d4ed8);
            height: 100%;
            border-radius: 4px;
            transition: width 0.3s ease;
        }
        
        .card {
            background: white;
            border-radius: 12px;
            padding: 30px;
            box-shadow: 0 4px 6px -1px rgba(0, 0, 0, 0.1);
            margin-bottom: 20px;
        }
        
        .card h2 {
            color: #1e293b;
            margin-bottom: 20px;
            font-size: 1.5rem;
        }
        
        .form-group {
            margin-bottom: 20px;
        }
        
        .form-group label {
            display: block;
            margin-bottom: 8px;
            font-weight: 500;
            color: #374151;
        }
        
        .form-group input {
            width: 100%;
            padding: 12px 16px;
            border: 1px solid #d1d5db;
            border-radius: 8px;
            font-size: 16px;
            transition: border-color 0.2s;
        }
        
        .form-group input:focus {
            outline: none;
            border-color: #3b82f6;
            box-shadow: 0 0 0 3px rgba(59, 130, 246, 0.1);
        }
        
        .form-group .hint {
            font-size: 0.875rem;
            color: #6b7280;
            margin-top: 4px;
        }
        
        .btn {
            display: inline-block;
            padding: 12px 24px;
            background: #3b82f6;
            color: white;
            text-decoration: none;
            border-radius: 8px;
            font-weight: 500;
            border: none;
            cursor: pointer;
            font-size: 16px;
            transition: background-color 0.2s;
        }
        
        .btn:hover {
            background: #2563eb;
        }
        
        .btn-secondary {
            background: #6b7280;
        }
        
        .btn-secondary:hover {
            background: #4b5563;
        }
        
        .btn-success {
            background: #10b981;
        }
        
        .btn-success:hover {
            background: #059669;
        }
        
        .requirements-list {
            list-style: none;
        }
        
        .requirements-list li {
            display: flex;
            align-items: center;
            padding: 12px 0;
            border-bottom: 1px solid #f1f5f9;
        }
        
        .requirements-list li:last-child {
            border-bottom: none;
        }
        
        .requirement-status {
            width: 24px;
            height: 24px;
            border-radius: 50%;
            display: flex;
            align-items: center;
            justify-content: center;
            margin-right: 12px;
            font-size: 14px;
            font-weight: bold;
            color: white;
        }
        
        .status-passed {
            background: #10b981;
        }
        
        .status-failed {
            background: #ef4444;
        }
        
        .error-message {
            background: #fef2f2;
            border: 1px solid #fecaca;
            color: #dc2626;
            padding: 12px 16px;
            border-radius: 8px;
            margin-bottom: 20px;
        }
        
        .success-message {
            background: #f0fdf4;
            border: 1px solid #bbf7d0;
            color: #16a34a;
            padding: 12px 16px;
            border-radius: 8px;
            margin-bottom: 20px;
        }
        
        .step-indicator {
            display: flex;
            justify-content: center;
            margin-bottom: 30px;
        }
        
        .step-item {
            display: flex;
            align-items: center;
        }
        
        .step-number {
            width: 32px;
            height: 32px;
            border-radius: 50%;
            display: flex;
            align-items: center;
            justify-content: center;
            font-weight: bold;
            margin-right: 8px;
        }
        
        .step-active .step-number {
            background: #3b82f6;
            color: white;
        }
        
        .step-inactive .step-number {
            background: #e2e8f0;
            color: #64748b;
        }
        
        .step-completed .step-number {
            background: #10b981;
            color: white;
        }
        
        .step-label {
            color: #64748b;
            font-size: 0.875rem;
        }
        
        .step-divider {
            width: 40px;
            height: 2px;
            background: #e2e8f0;
            margin: 0 16px;
        }
        
        @media (max-width: 640px) {
            .container {
                padding: 20px 16px;
            }
            
            .header h1 {
                font-size: 2rem;
            }
            
            .card {
                padding: 20px;
            }
            
            .step-indicator {
                flex-wrap: wrap;
                gap: 16px;
            }
            
            .step-divider {
                display: none;
            }
        }
    </style>
</head>
<body>
    <div class="container">
        <div class="header">
            <h1>AI Visibility Tracker</h1>
            <p>Installation Wizard</p>
        </div>
        
        <div class="progress-bar">
            <div class="progress-fill" style="width: <?php echo ($step / 4) * 100; ?>%"></div>
        </div>
        
        <div class="step-indicator">
            <div class="step-item <?php echo $step >= 1 ? 'step-active' : 'step-inactive'; ?>">
                <div class="step-number">1</div>
                <div class="step-label">Requirements</div>
            </div>
            <div class="step-divider"></div>
            <div class="step-item <?php echo $step >= 2 ? 'step-active' : 'step-inactive'; ?>">
                <div class="step-number">2</div>
                <div class="step-label">Database</div>
            </div>
            <div class="step-divider"></div>
            <div class="step-item <?php echo $step >= 3 ? 'step-active' : 'step-inactive'; ?>">
                <div class="step-number">3</div>
                <div class="step-label">Configuration</div>
            </div>
            <div class="step-divider"></div>
            <div class="step-item <?php echo $step >= 4 ? 'step-active' : 'step-inactive'; ?>">
                <div class="step-number">4</div>
                <div class="step-label">Install</div>
            </div>
        </div>
        
        <?php if (!empty($errors)): ?>
            <div class="error-message">
                <?php foreach ($errors as $error): ?>
                    <div><?php echo htmlspecialchars($error); ?></div>
                <?php endforeach; ?>
            </div>
        <?php endif; ?>
        
        <?php if ($success): ?>
            <div class="success-message">
                Installation completed successfully! You can now log in with your admin credentials.
            </div>
            <div class="card">
                <h2>Next Steps</h2>
                <ul style="margin-left: 20px; margin-bottom: 20px;">
                    <li>Delete the <code>/install</code> directory for security</li>
                    <li>Log in to your admin dashboard</li>
                    <li>Configure your AI model API keys</li>
                    <li>Set up rate limits and prompts</li>
                    <li>Start tracking your website's AI visibility!</li>
                </ul>
                <a href="/admin" class="btn btn-success">Go to Admin Dashboard</a>
            </div>
        <?php else: ?>
            <?php if ($step == 1): ?>
                <div class="card">
                    <h2>System Requirements</h2>
                    <p>Please ensure your server meets the following requirements:</p>
                    <ul class="requirements-list">
                        <?php foreach ($requirements as $key => $req): ?>
                            <li>
                                <div class="requirement-status <?php echo $req['passed'] ? 'status-passed' : 'status-failed'; ?>">
                                    <?php echo $req['passed'] ? '✓' : '✗'; ?>
                                </div>
                                <div style="flex: 1;">
                                    <strong><?php echo htmlspecialchars($req['name'] ?? ucwords(str_replace('_', ' ', $key))); ?></strong>
                                    <div style="font-size: 0.875rem; color: #6b7280; margin-bottom: 4px;">
                                        <?php if ($key === 'php_version'): ?>
                                            Required: <?php echo $req['required']; ?>, Current: <?php echo $req['current']; ?>
                                        <?php else: ?>
                                            <?php echo $req['current'] ? 'Available' : 'Not Available'; ?>
                                        <?php endif; ?>
                                    </div>
                                    <?php if (!$req['passed']): ?>
                                        <div class="requirement-error" style="font-size: 0.8rem; color: #dc2626; background: #fef2f2; padding: 8px; border-radius: 4px; margin-top: 4px;">
                                            <strong>Fix:</strong> <?php echo htmlspecialchars($req['message']); ?>
                                        </div>
                                    <?php else: ?>
                                        <div style="font-size: 0.8rem; color: #059669; margin-top: 4px;">
                                            <?php echo htmlspecialchars($req['message']); ?>
                                        </div>
                                    <?php endif; ?>
                                </div>
                            </li>
                        <?php endforeach; ?>
                    </ul>
                    
                    <?php if ($allRequirementsMet): ?>
                        <a href="?step=2" class="btn" style="margin-top: 20px;">Continue to Database Setup</a>
                    <?php else: ?>
                        <div style="margin-top: 20px; color: #dc2626;">
                            Please fix the requirements above before continuing.
                        </div>
                    <?php endif; ?>
                </div>
            <?php elseif ($step == 2): ?>
                <div class="card">
                    <h2>Database Configuration</h2>
                    <form method="POST" action="?step=2">
                        <div class="form-group">
                            <label for="db_host">Database Host</label>
                            <input type="text" id="db_host" name="db_host" value="localhost" required>
                            <div class="hint">Usually localhost or 127.0.0.1</div>
                        </div>
                        
                        <div class="form-group">
                            <label for="db_name">Database Name</label>
                            <input type="text" id="db_name" name="db_name" required>
                            <div class="hint">The database will be created if it doesn't exist</div>
                        </div>
                        
                        <div class="form-group">
                            <label for="db_user">Database Username</label>
                            <input type="text" id="db_user" name="db_user" required>
                        </div>
                        
                        <div class="form-group">
                            <label for="db_pass">Database Password</label>
                            <input type="password" id="db_pass" name="db_pass">
                        </div>
                        
                        <button type="submit" class="btn">Test Connection & Continue</button>
                    </form>
                </div>
            <?php elseif ($step == 3): ?>
                <div class="card">
                    <h2>Application Configuration</h2>
                    <form method="POST" action="?step=3">
                        <div class="form-group">
                            <label for="site_name">Site Name</label>
                            <input type="text" id="site_name" name="site_name" value="AI Visibility Tracker" required>
                        </div>
                        
                        <div class="form-group">
                            <label for="base_url">Base URL</label>
                            <input type="url" id="base_url" name="base_url" value="<?php echo 'https://' . $_SERVER['HTTP_HOST']; ?>" required>
                            <div class="hint">Include https:// and without trailing slash</div>
                        </div>
                        
                        <div class="form-group">
                            <label for="admin_email">Admin Email</label>
                            <input type="email" id="admin_email" name="admin_email" required>
                        </div>
                        
                        <div class="form-group">
                            <label for="admin_password">Admin Password</label>
                            <input type="password" id="admin_password" name="admin_password" required>
                            <div class="hint">Minimum 8 characters</div>
                        </div>
                        
                        <div class="form-group">
                            <label for="confirm_password">Confirm Password</label>
                            <input type="password" id="confirm_password" name="confirm_password" required>
                        </div>
                        
                        <div class="form-group">
                            <label for="openrouter_api_key">OpenRouter API Key</label>
                            <input type="text" id="openrouter_api_key" name="openrouter_api_key">
                            <div class="hint">Optional for now, can be configured later in admin panel</div>
                        </div>
                        
                        <button type="submit" class="btn">Continue to Installation</button>
                    </form>
                </div>
            <?php elseif ($step == 4): ?>
                <div class="card">
                    <h2>Installation Progress</h2>
                    <p>Installing AI Visibility Tracker...</p>
                    <div style="text-align: center; margin: 40px 0;">
                        <div style="width: 40px; height: 40px; border: 3px solid #e2e8f0; border-top-color: #3b82f6; border-radius: 50%; animation: spin 1s linear infinite; margin: 0 auto;"></div>
                        <p style="margin-top: 16px; color: #64748b;">Please wait while we set up your application...</p>
                    </div>
                    
                    <form method="POST" action="?step=4" style="display: none;">
                        <button type="submit" class="btn">Complete Installation</button>
                    </form>
                    
                    <script>
                        setTimeout(function() {
                            document.querySelector('form').submit();
                        }, 2000);
                    </script>
                </div>
            <?php endif; ?>
        <?php endif; ?>
    </div>
    
    <style>
        @keyframes spin {
            to { transform: rotate(360deg); }
        }
    </style>
</body>
</html>