# AI Visibility Tracker

A production-ready SaaS web application for tracking AI visibility across major AI models. Monitor how AI models perceive and recommend your website.

## 🚀 Features

- **AI Request Engine**: Queue-based system for testing AI model recognition and recommendations
- **Multi-Model Support**: GPT-4, GPT-3.5, Claude 3, Gemini Pro integration via OpenRouter
- **Analytics Dashboard**: Track visibility scores, mentions, and recommendations over time
- **User Management**: Complete user system with roles, authentication, and authorization
- **Admin Panel**: Full control over users, websites, AI models, and system settings
- **WordPress-like Installer**: One-click installation process
- **Rate Limiting**: Built-in protection against abuse
- **Security**: CSRF protection, XSS prevention, SQL injection protection
- **Logging**: Comprehensive logging system for debugging and monitoring

## 📋 Requirements

- PHP 8.0 or higher
- MySQL 5.7 or higher
- Apache or Nginx web server
- PDO MySQL extension
- cURL extension
- JSON extension
- Write permissions for /config, /logs, /public/uploads

## 🛠️ Installation

### Method 1: WordPress-like Installer (Recommended)

1. **Upload Files**
   - Upload the entire application to your web server
   - Ensure the web root points to the `/public` directory

2. **Set Permissions**
   ```bash
   chmod 755 /path/to/ai-visibility-tracker/logs
   chmod 755 /path/to/ai-visibility-tracker/public/uploads
   ```

3. **Run Installer**
   - Visit `https://yourdomain.com/install` in your browser
   - Follow the step-by-step installation wizard
   - Provide database credentials, site settings, and admin account details
   - The installer will automatically create the database and configuration

4. **Complete Setup**
   - Delete the `/install` directory after installation
   - Log in to your admin dashboard
   - Configure your OpenRouter API key
   - Start tracking websites!

### Method 2: Manual Installation

1. **Create Database**
   ```sql
   CREATE DATABASE ai_visibility_tracker CHARACTER SET utf8mb4 COLLATE utf8mb4_unicode_ci;
   ```

2. **Import Schema**
   ```bash
   mysql -u username -p ai_visibility_tracker < install/schema.sql
   ```

3. **Create Config File**
   ```php
   <?php
   define('DB_HOST', 'localhost');
   define('DB_NAME', 'ai_visibility_tracker');
   define('DB_USER', 'username');
   define('DB_PASS', 'password');
   define('BASE_URL', 'https://yourdomain.com');
   define('SITE_NAME', 'AI Visibility Tracker');
   define('SECRET_KEY', 'your-secret-key-here');
   define('DEBUG_MODE', false);
   define('OPENROUTER_API_KEY', 'your-openrouter-api-key');
   ?>
   ```

4. **Set Permissions**
   ```bash
   chmod 644 config.php
   chmod 755 logs/
   chmod 755 public/uploads/
   ```

## 🔧 Configuration

### Environment Variables

The following constants can be configured in `config.php`:

- `DB_HOST` - Database host
- `DB_NAME` - Database name
- `DB_USER` - Database username
- `DB_PASS` - Database password
- `BASE_URL` - Your website URL (without trailing slash)
- `SITE_NAME` - Your site name
- `SECRET_KEY` - Random secret key for security
- `DEBUG_MODE` - Enable debug logging (true/false)
- `LOG_LEVEL` - Logging level (0-7, 7 = debug)
- `OPENROUTER_API_KEY` - Your OpenRouter API key

### Rate Limiting

Default rate limits per user:
- 60 requests per minute
- 1000 requests per hour

Configure per-model limits in the admin panel.

## 🚀 Running the Application

### Web Server Setup

#### Apache
Create a `.htaccess` file in the `/public` directory:
```apache
RewriteEngine On
RewriteCond %{REQUEST_FILENAME} !-f
RewriteCond %{REQUEST_FILENAME} !-d
RewriteRule ^(.*)$ index.php [QSA,L]
```

#### Nginx
Add to your server block:
```nginx
location / {
    try_files $uri $uri/ /index.php?$query_string;
}

location ~ \.php$ {
    fastcgi_pass unix:/var/run/php/php8.0-fpm.sock;
    fastcgi_index index.php;
    include fastcgi_params;
}
```

### Queue Worker

The queue worker processes AI requests in the background:

```bash
# Run worker manually
php worker.php

# Run with supervisor or systemd for production
# See deployment section below
```

### Cron Jobs

Set up periodic tasks:

```bash
# Run every hour to scan websites and clean up
0 * * * * /usr/bin/php /path/to/ai-visibility-tracker/cron.php
```

## 🔐 Security Features

- **Prepared Statements**: All database queries use prepared statements
- **Password Hashing**: bcrypt with cost factor 12
- **CSRF Protection**: All forms include CSRF tokens
- **XSS Protection**: Input sanitization and output escaping
- **Rate Limiting**: Per-user and per-model request limits
- **Brute Force Protection**: Login attempt limiting
- **Session Security**: HttpOnly, Secure, SameSite cookies
- **File Upload Limits**: Restricted file types and sizes

## 📊 Architecture

```
ai-visibility-tracker/
├── app/
│   ├── controllers/     # Application controllers
│   ├── models/          # Data models
│   ├── services/        # Business logic services
│   └── views/           # Template files
├── core/                # Core framework classes
│   ├── Database.php     # Database abstraction
│   ├── Auth.php         # Authentication system
│   ├── Queue.php        # Job queue system
│   └── Logger.php       # Logging system
├── install/             # Installation wizard
├── public/              # Web accessible files
│   ├── css/             # Stylesheets
│   ├── js/              # JavaScript files
│   └── uploads/         # User uploads
├── logs/                # Application logs
├── config.php           # Application configuration
├── worker.php           # Queue worker script
├── cron.php             # Periodic tasks
└── public/index.php     # Application entry point
```

## 🔍 Monitoring & Debugging

### Logs

Application logs are stored in `/logs/` directory:
- Structured JSON format
- Daily rotation
- 30-day retention
- Levels: EMERGENCY, ALERT, CRITICAL, ERROR, WARNING, NOTICE, INFO, DEBUG

### Admin Dashboard

Access comprehensive analytics:
- User statistics
- Website tracking status
- Queue performance
- Model usage metrics
- Error rates
- System logs

### Health Checks

Monitor application health:
- Database connectivity
- Queue status
- API response times
- Error rates

## 🚀 Production Deployment

### Server Requirements

- **PHP**: 8.0+ with FPM
- **Database**: MySQL 5.7+ or MariaDB 10.3+
- **Web Server**: Nginx (recommended) or Apache
- **Memory**: Minimum 512MB RAM
- **Storage**: 1GB+ for logs and uploads

### Process Management

#### Using Supervisor

Install supervisor:
```bash
sudo apt-get install supervisor
```

Create config `/etc/supervisor/conf.d/ai-visibility-worker.conf`:
```ini
[program:ai-visibility-worker]
process_name=%(program_name)s_%(process_num)02d
command=/usr/bin/php /path/to/ai-visibility-tracker/worker.php %(process_num)02d
autostart=true
autorestart=true
user=www-data
numprocs=2
redirect_stderr=true
stdout_logfile=/var/log/ai-visibility-worker.log
```

Start workers:
```bash
sudo supervisorctl reread
sudo supervisorctl update
sudo supervisorctl start ai-visibility-worker:*
```

#### Using Systemd

Create service `/etc/systemd/system/ai-visibility-worker.service`:
```ini
[Unit]
Description=AI Visibility Tracker Queue Worker
After=network.target

[Service]
Type=simple
User=www-data
WorkingDirectory=/path/to/ai-visibility-tracker
ExecStart=/usr/bin/php worker.php
Restart=always
RestartSec=10

[Install]
WantedBy=multi-user.target
```

Enable and start:
```bash
sudo systemctl enable ai-visibility-worker
sudo systemctl start ai-visibility-worker
```

### SSL/TLS

Always use HTTPS in production:
```nginx
server {
    listen 80;
    server_name yourdomain.com;
    return 301 https://$server_name$request_uri;
}

server {
    listen 443 ssl http2;
    server_name yourdomain.com;
    
    ssl_certificate /path/to/cert.pem;
    ssl_certificate_key /path/to/key.pem;
    
    # Security headers
    add_header X-Frame-Options DENY;
    add_header X-Content-Type-Options nosniff;
    add_header X-XSS-Protection "1; mode=block";
    add_header Strict-Transport-Security "max-age=31536000; includeSubDomains";
}
```

### Database Optimization

For high-traffic installations:
```sql
-- Add indexes for better performance
CREATE INDEX idx_analytics_composite ON analytics(website_id, model_id, recorded_at);
CREATE INDEX idx_job_queue_composite ON job_queue(status, priority, created_at);

-- Partition analytics table for large datasets
-- See documentation for partitioning setup
```

### Caching

Enable OPcache for better performance:
```ini
opcache.enable=1
opcache.memory_consumption=128
opcache.max_accelerated_files=4000
opcache.revalidate_freq=60
```

## 🔧 API Integration

### OpenRouter Setup

1. Sign up at [OpenRouter](https://openrouter.ai/)
2. Generate an API key
3. Add to config.php: `define('OPENROUTER_API_KEY', 'your-key-here');`
4. Configure rate limits in admin panel

### Adding New AI Models

1. Add model to `ai_models` table
2. Configure API endpoint and rate limits
3. Update prompt templates if needed
4. Test with sample requests

## 📈 Scaling

### Horizontal Scaling

- Run multiple queue workers (scale workers based on load)
- Use database connection pooling
- Implement Redis for caching (optional)
- Load balance web servers

### Vertical Scaling

- Increase PHP memory limit
- Optimize database queries
- Use faster storage (SSD/NVMe)
- More CPU cores for parallel processing

### Monitoring

Monitor these metrics:
- Queue backlog size
- Average job processing time
- Database query performance
- Memory usage
- Error rates
- User activity

## 🐛 Troubleshooting

### Common Issues

**Installation fails**
- Check PHP version (8.0+ required)
- Verify database credentials
- Ensure write permissions on logs/ and uploads/
- Check error logs for specific errors

**Queue worker not processing jobs**
- Verify database connection
- Check for failed jobs in queue
- Review error logs
- Ensure rate limits not exceeded

**AI requests failing**
- Verify OpenRouter API key
- Check rate limits
- Review API response logs
- Ensure model is active

**Slow performance**
- Check database indexes
- Review slow query log
- Consider adding more workers
- Enable OPcache

### Debug Mode

Enable debug mode in config.php:
```php
define('DEBUG_MODE', true);
define('LOG_LEVEL', 7); // Debug level
```

This will:
- Show detailed error messages
- Log all database queries
- Enable verbose logging
- Display stack traces

**Remember to disable debug mode in production!**

## 🤝 Contributing

1. Fork the repository
2. Create a feature branch
3. Make your changes
4. Add tests if applicable
5. Submit a pull request

## 📄 License

This software is proprietary. All rights reserved.

## 🆘 Support

For support:
- Check the troubleshooting section
- Review application logs
- Contact your system administrator
- Check the admin dashboard for system status

---

**AI Visibility Tracker** - Monitor your AI presence across the digital landscape.