<?php
/**
 * AI Visibility Tracker - Cron Jobs
 * 
 * Run this script periodically (e.g., every hour) via cron
 * Usage: php cron.php
 */

// Start session
session_start();

// Define base path
define('BASE_PATH', __DIR__);

// Check if application is installed
if (!file_exists(BASE_PATH . '/config.php')) {
    die("Application is not installed. Please run the installer first.\n");
}

// Load configuration
require_once BASE_PATH . '/config.php';

// Load core classes
require_once BASE_PATH . '/core/Logger.php';
require_once BASE_PATH . '/core/Database.php';
require_once BASE_PATH . '/core/Queue.php';

// Load models
require_once BASE_PATH . '/app/models/Website.php';

// Load services
require_once BASE_PATH . '/app/services/AIService.php';

// Initialize logger
$logger = Logger::getInstance();

$logger->logInfo("Cron job started");

try {
    $db = Database::getInstance();
    $aiService = new AIService();
    $websiteModel = new Website();
    
    // Get websites that need scanning (older than 24 hours)
    $websites = $websiteModel->getWebsitesNeedingScan(24);
    
    $logger->logInfo("Found websites needing scan", ['count' => count($websites)]);
    
    foreach ($websites as $website) {
        try {
            // Get all active models
            $models = $db->fetchAll(
                "SELECT id FROM ai_models WHERE is_active = 1"
            );
            
            // Queue tests for each model
            foreach ($models as $model) {
                $aiService->queueRecognitionTest($website['id'], $website['user_id'], $model['id']);
                $aiService->queueQuestionAssociation($website['id'], $website['user_id'], $model['id']);
            }
            
            // Update last scan time
            $websiteModel->updateLastScan($website['id']);
            
            $logger->logInfo("Queued scan for website", [
                'website_id' => $website['id'],
                'models' => count($models)
            ]);
            
            // Small delay to avoid overwhelming the queue
            usleep(100000); // 100ms
            
        } catch (Exception $e) {
            $logger->logError("Failed to queue scan for website", [
                'website_id' => $website['id'],
                'error' => $e->getMessage()
            ]);
        }
    }
    
    // Clean up old logs
    $logger->clearOldLogs(30); // Keep logs for 30 days
    
    // Clean up old rate limit logs
    $rateLimiter = new RateLimiterService();
    $rateLimiter->clearOldLogs(30);
    
    // Clean up old completed jobs
    $queue = Queue::getInstance();
    $queue->clearCompletedJobs(7); // Keep completed jobs for 7 days
    
    // Clean up old login attempts
    $db->execute(
        "DELETE FROM login_attempts WHERE attempt_time < DATE_SUB(NOW(), INTERVAL 1 DAY)"
    );
    
    // Clean up expired password resets
    $db->execute(
        "DELETE FROM password_resets WHERE expires_at < NOW()"
    );
    
    $logger->logInfo("Cron job completed successfully");
    
} catch (Exception $e) {
    $logger->logError("Cron job failed", [
        'error' => $e->getMessage(),
        'trace' => $e->getTraceAsString()
    ]);
    
    throw $e;
}

echo "Cron job completed at " . date('Y-m-d H:i:s') . "\n";