<?php

class AdminController extends BaseController
{
    private $userModel;
    private $websiteModel;

    public function __construct()
    {
        parent::__construct();
        $this->auth->requireAdmin();
        $this->userModel = new User();
        $this->websiteModel = new Website();
    }

    public function index()
    {
        $userStats = $this->userModel->getStats();
        $websiteStats = $this->websiteModel->getGlobalStats();
        
        $queueStats = $this->db->fetch(
            "SELECT 
                SUM(CASE WHEN status = 'pending' THEN 1 ELSE 0 END) as pending,
                SUM(CASE WHEN status = 'processing' THEN 1 ELSE 0 END) as processing,
                SUM(CASE WHEN status = 'completed' THEN 1 ELSE 0 END) as completed,
                SUM(CASE WHEN status = 'failed' THEN 1 ELSE 0 END) as failed
             FROM job_queue"
        );
        
        $recentUsers = $this->db->fetchAll(
            "SELECT id, email, role, is_active, created_at FROM users ORDER BY created_at DESC LIMIT 10"
        );
        
        $recentWebsites = $this->db->fetchAll(
            "SELECT w.*, u.email as user_email 
             FROM websites w 
             JOIN users u ON w.user_id = u.id 
             ORDER BY w.created_at DESC 
             LIMIT 10"
        );
        
        echo $this->renderLayout('admin.index', [
            'page_title' => 'Admin Dashboard - AI Visibility Tracker',
            'user_stats' => $userStats,
            'website_stats' => $websiteStats,
            'queue_stats' => $queueStats,
            'recent_users' => $recentUsers,
            'recent_websites' => $recentWebsites
        ]);
    }

    public function users()
    {
        $page = (int)($_GET['page'] ?? 1);
        $perPage = 20;
        $offset = ($page - 1) * $perPage;
        
        $search = $_GET['search'] ?? '';
        
        if (!empty($search)) {
            $users = $this->userModel->search($search, $perPage);
            $totalCount = count($users);
        } else {
            $users = $this->userModel->getAll($perPage, $offset);
            $totalCount = $this->db->fetch("SELECT COUNT(*) as count FROM users")['count'];
        }
        
        $pagination = $this->paginate($totalCount, $page, $perPage, '/admin/users');
        
        echo $this->renderLayout('admin.users', [
            'page_title' => 'Users - Admin - AI Visibility Tracker',
            'users' => $users,
            'pagination' => $pagination,
            'search' => $search
        ]);
    }

    public function editUser($userId)
    {
        $user = $this->userModel->getById($userId);
        
        if (!$user) {
            $this->redirect('/admin/users');
        }
        
        $errors = [];
        $success = false;
        
        if ($_SERVER['REQUEST_METHOD'] === 'POST') {
            $this->validateCsrf();
            
            $email = $this->sanitizeInput($_POST['email'] ?? '');
            $role = $this->sanitizeInput($_POST['role'] ?? 'member');
            $isActive = isset($_POST['is_active']) ? 1 : 0;
            
            $validationErrors = $this->validateInput($_POST, [
                'email' => 'required|email',
                'role' => 'required|in:admin,member'
            ]);
            
            if (empty($validationErrors)) {
                try {
                    $this->userModel->update($userId, [
                        'email' => $email,
                        'role' => $role,
                        'is_active' => $isActive
                    ]);
                    
                    $success = true;
                    $this->logger->logInfo("User updated by admin", ['user_id' => $userId, 'admin_id' => $this->auth->getUserId()]);
                    
                } catch (Exception $e) {
                    $errors[] = $e->getMessage();
                }
            } else {
                $errors = array_merge($errors, array_flatten($validationErrors));
            }
        }
        
        $csrfToken = $this->auth->getCsrfToken();
        
        echo $this->renderLayout('admin.edit-user', [
            'page_title' => 'Edit User - Admin - AI Visibility Tracker',
            'user' => $user,
            'errors' => $errors,
            'success' => $success,
            'csrf_token' => $csrfToken
        ]);
    }

    public function resetUserPassword($userId)
    {
        if ($_SERVER['REQUEST_METHOD'] !== 'POST') {
            $this->redirect('/admin/users');
        }
        
        $this->validateCsrf();
        
        $newPassword = bin2hex(random_bytes(8));
        
        try {
            $this->userModel->updatePassword($userId, $newPassword);
            
            $this->logger->logInfo("Password reset by admin", [
                'user_id' => $userId,
                'admin_id' => $this->auth->getUserId()
            ]);
            
            $_SESSION['flash_message'] = 'Password reset successfully. New password: ' . $newPassword;
            
        } catch (Exception $e) {
            $_SESSION['flash_error'] = 'Failed to reset password: ' . $e->getMessage();
        }
        
        $this->redirect('/admin/users/' . $userId);
    }

    public function websites()
    {
        $page = (int)($_GET['page'] ?? 1);
        $perPage = 20;
        $offset = ($page - 1) * $perPage;
        
        $search = $_GET['search'] ?? '';
        
        if (!empty($search)) {
            $websites = $this->websiteModel->search($search, $perPage);
            $totalCount = count($websites);
        } else {
            $websites = $this->websiteModel->getAll($perPage, $offset);
            $totalCount = $this->db->fetch("SELECT COUNT(*) as count FROM websites")['count'];
        }
        
        $pagination = $this->paginate($totalCount, $page, $perPage, '/admin/websites');
        
        echo $this->renderLayout('admin.websites', [
            'page_title' => 'Websites - Admin - AI Visibility Tracker',
            'websites' => $websites,
            'pagination' => $pagination,
            'search' => $search
        ]);
    }

    public function viewWebsite($websiteId)
    {
        $website = $this->websiteModel->getById($websiteId);
        
        if (!$website) {
            $this->redirect('/admin/websites');
        }
        
        $analytics = $this->db->fetchAll(
            "SELECT a.*, m.display_name as model_name
             FROM analytics a
             JOIN ai_models m ON a.model_id = m.id
             WHERE a.website_id = ?
             ORDER BY a.recorded_at DESC
             LIMIT 50",
            [$websiteId]
        );
        
        $jobs = $this->db->fetchAll(
            "SELECT j.*, m.display_name as model_name
             FROM job_queue j
             JOIN ai_models m ON JSON_UNQUOTE(JSON_EXTRACT(j.payload, '$.model_id')) = m.id
             WHERE j.website_id = ?
             ORDER BY j.created_at DESC
             LIMIT 20",
            [$websiteId]
        );
        
        echo $this->renderLayout('admin.view-website', [
            'page_title' => 'View Website - Admin - AI Visibility Tracker',
            'website' => $website,
            'analytics' => $analytics,
            'jobs' => $jobs
        ]);
    }

    public function forceRescan($websiteId)
    {
        if ($_SERVER['REQUEST_METHOD'] !== 'POST') {
            $this->redirect('/admin/websites/' . $websiteId);
        }
        
        $this->validateCsrf();
        
        $website = $this->websiteModel->getById($websiteId);
        
        if (!$website) {
            $_SESSION['flash_error'] = 'Website not found';
            $this->redirect('/admin/websites');
        }
        
        try {
            $models = $this->db->fetchAll(
                "SELECT id FROM ai_models WHERE is_active = 1"
            );
            
            $aiService = new AIService();
            
            foreach ($models as $model) {
                $aiService->queueRecognitionTest($websiteId, $website['user_id'], $model['id']);
                $aiService->queueQuestionAssociation($websiteId, $website['user_id'], $model['id']);
            }
            
            $this->logger->logInfo("Force rescan initiated", [
                'website_id' => $websiteId,
                'admin_id' => $this->auth->getUserId()
            ]);
            
            $_SESSION['flash_message'] = 'Rescan initiated successfully';
            
        } catch (Exception $e) {
            $_SESSION['flash_error'] = 'Failed to initiate rescan: ' . $e->getMessage();
        }
        
        $this->redirect('/admin/websites/' . $websiteId);
    }

    public function models()
    {
        $models = $this->db->fetchAll(
            "SELECT m.*, 
                    COUNT(r.id) as total_requests,
                    AVG(r.processing_time_ms) as avg_processing_time
             FROM ai_models m
             LEFT JOIN ai_requests r ON m.id = r.model_id
             GROUP BY m.id
             ORDER BY m.display_name"
        );
        
        echo $this->renderLayout('admin.models', [
            'page_title' => 'AI Models - Admin - AI Visibility Tracker',
            'models' => $models
        ]);
    }

    public function editModel($modelId)
    {
        $model = $this->db->fetch(
            "SELECT * FROM ai_models WHERE id = ?",
            [$modelId]
        );
        
        if (!$model) {
            $this->redirect('/admin/models');
        }
        
        $errors = [];
        $success = false;
        
        if ($_SERVER['REQUEST_METHOD'] === 'POST') {
            $this->validateCsrf();
            
            $isActive = isset($_POST['is_active']) ? 1 : 0;
            $maxRequestsPerMinute = (int)($_POST['max_requests_per_minute'] ?? 60);
            $maxRequestsPerHour = (int)($_POST['max_requests_per_hour'] ?? 1000);
            
            try {
                $this->db->execute(
                    "UPDATE ai_models SET 
                     is_active = ?, 
                     max_requests_per_minute = ?, 
                     max_requests_per_hour = ?,
                     updated_at = NOW()
                     WHERE id = ?",
                    [$isActive, $maxRequestsPerMinute, $maxRequestsPerHour, $modelId]
                );
                
                $success = true;
                $this->logger->logInfo("AI model updated", ['model_id' => $modelId, 'admin_id' => $this->auth->getUserId()]);
                
            } catch (Exception $e) {
                $errors[] = $e->getMessage();
            }
        }
        
        $csrfToken = $this->auth->getCsrfToken();
        
        echo $this->renderLayout('admin.edit-model', [
            'page_title' => 'Edit AI Model - Admin - AI Visibility Tracker',
            'model' => $model,
            'errors' => $errors,
            'success' => $success,
            'csrf_token' => $csrfToken
        ]);
    }

    public function prompts()
    {
        $prompts = $this->db->fetchAll(
            "SELECT * FROM prompt_templates ORDER BY request_type, version DESC"
        );
        
        echo $this->renderLayout('admin.prompts', [
            'page_title' => 'Prompt Templates - Admin - AI Visibility Tracker',
            'prompts' => $prompts
        ]);
    }

    public function editPrompt($promptId)
    {
        $prompt = $this->db->fetch(
            "SELECT * FROM prompt_templates WHERE id = ?",
            [$promptId]
        );
        
        if (!$prompt) {
            $this->redirect('/admin/prompts');
        }
        
        $errors = [];
        $success = false;
        
        if ($_SERVER['REQUEST_METHOD'] === 'POST') {
            $this->validateCsrf();
            
            $template = $_POST['template'] ?? '';
            $isActive = isset($_POST['is_active']) ? 1 : 0;
            
            try {
                $this->db->execute(
                    "UPDATE prompt_templates SET 
                     template = ?, 
                     is_active = ?,
                     updated_at = NOW()
                     WHERE id = ?",
                    [$template, $isActive, $promptId]
                );
                
                $success = true;
                $this->logger->logInfo("Prompt template updated", ['prompt_id' => $promptId, 'admin_id' => $this->auth->getUserId()]);
                
            } catch (Exception $e) {
                $errors[] = $e->getMessage();
            }
        }
        
        $csrfToken = $this->auth->getCsrfToken();
        
        echo $this->renderLayout('admin.edit-prompt', [
            'page_title' => 'Edit Prompt - Admin - AI Visibility Tracker',
            'prompt' => $prompt,
            'errors' => $errors,
            'success' => $success,
            'csrf_token' => $csrfToken
        ]);
    }

    public function settings()
    {
        $settings = $this->db->fetchAll(
            "SELECT * FROM system_settings ORDER BY setting_key"
        );
        
        $errors = [];
        $success = false;
        
        if ($_SERVER['REQUEST_METHOD'] === 'POST') {
            $this->validateCsrf();
            
            try {
                foreach ($settings as $setting) {
                    $key = $setting['setting_key'];
                    $value = $_POST[$key] ?? '';
                    
                    $this->db->execute(
                        "UPDATE system_settings SET setting_value = ? WHERE setting_key = ?",
                        [$value, $key]
                    );
                }
                
                $success = true;
                $this->logger->logInfo("System settings updated", ['admin_id' => $this->auth->getUserId()]);
                
            } catch (Exception $e) {
                $errors[] = $e->getMessage();
            }
        }
        
        $csrfToken = $this->auth->getCsrfToken();
        
        echo $this->renderLayout('admin.settings', [
            'page_title' => 'Settings - Admin - AI Visibility Tracker',
            'settings' => $settings,
            'errors' => $errors,
            'success' => $success,
            'csrf_token' => $csrfToken
        ]);
    }

    public function logs()
    {
        $date = $_GET['date'] ?? date('Y-m-d');
        $level = $_GET['level'] ?? '';
        $page = (int)($_GET['page'] ?? 1);
        $perPage = 100;
        
        $logFile = dirname(__DIR__, 2) . '/logs/' . $date . '.log';
        
        $logs = [];
        if (file_exists($logFile)) {
            $lines = file($logFile, FILE_IGNORE_NEW_LINES | FILE_SKIP_EMPTY_LINES);
            $lines = array_reverse($lines);
            
            foreach ($lines as $line) {
                $log = json_decode($line, true);
                if ($log) {
                    if (empty($level) || $log['level'] === $level) {
                        $logs[] = $log;
                    }
                }
            }
        }
        
        $totalLogs = count($logs);
        $offset = ($page - 1) * $perPage;
        $logs = array_slice($logs, $offset, $perPage);
        
        $pagination = $this->paginate($totalLogs, $page, $perPage, '/admin/logs');
        
        $availableDates = [];
        $logFiles = glob(dirname(__DIR__, 2) . '/logs/*.log');
        foreach ($logFiles as $file) {
            $availableDates[] = basename($file, '.log');
        }
        rsort($availableDates);
        
        echo $this->renderLayout('admin.logs', [
            'page_title' => 'Logs - Admin - AI Visibility Tracker',
            'logs' => $logs,
            'pagination' => $pagination,
            'date' => $date,
            'level' => $level,
            'available_dates' => $availableDates
        ]);
    }
}