<?php

class AuthController extends BaseController
{
    private $userModel;

    public function __construct()
    {
        parent::__construct();
        $this->userModel = new User();
    }

    public function login()
    {
        if ($this->auth->isLoggedIn()) {
            $this->redirect('/dashboard');
        }

        $errors = [];
        
        if ($_SERVER['REQUEST_METHOD'] === 'POST') {
            $this->validateCsrf();
            
            $email = $this->sanitizeInput($_POST['email'] ?? '');
            $password = $_POST['password'] ?? '';
            $remember = isset($_POST['remember']);
            
            $validationErrors = $this->validateInput($_POST, [
                'email' => 'required|email',
                'password' => 'required|min:6'
            ]);
            
            if (empty($validationErrors)) {
                try {
                    if ($this->auth->login($email, $password)) {
                        $redirectUrl = $_SESSION['redirect_after_login'] ?? '/dashboard';
                        unset($_SESSION['redirect_after_login']);
                        
                        $this->redirect($redirectUrl);
                    }
                } catch (Exception $e) {
                    $errors[] = $e->getMessage();
                    $this->logger->logWarning("Login failed", ['email' => $email, 'error' => $e->getMessage()]);
                }
            } else {
                $errors = array_merge($errors, array_flatten($validationErrors));
            }
        }
        
        $csrfToken = $this->auth->getCsrfToken();
        
        echo $this->renderLayout('auth.login', [
            'errors' => $errors,
            'csrf_token' => $csrfToken,
            'page_title' => 'Login'
        ]);
    }

    public function register()
    {
        if ($this->auth->isLoggedIn()) {
            $this->redirect('/dashboard');
        }

        $errors = [];
        $success = false;
        
        if ($_SERVER['REQUEST_METHOD'] === 'POST') {
            $this->validateCsrf();
            
            $email = $this->sanitizeInput($_POST['email'] ?? '');
            $password = $_POST['password'] ?? '';
            $confirmPassword = $_POST['confirm_password'] ?? '';
            
            $validationErrors = $this->validateInput($_POST, [
                'email' => 'required|email',
                'password' => 'required|min:8',
                'confirm_password' => 'required'
            ]);
            
            if (empty($validationErrors)) {
                if ($password !== $confirmPassword) {
                    $errors[] = 'Passwords do not match.';
                } else {
                    try {
                        $userId = $this->userModel->create($email, $password, 'member', true);
                        
                        $this->logger->logInfo("User registered", ['user_id' => $userId, 'email' => $email]);
                        
                        $this->auth->login($email, $password);
                        
                        $this->redirect('/dashboard');
                        
                    } catch (Exception $e) {
                        $errors[] = $e->getMessage();
                        $this->logger->logWarning("Registration failed", ['email' => $email, 'error' => $e->getMessage()]);
                    }
                }
            } else {
                $errors = array_merge($errors, array_flatten($validationErrors));
            }
        }
        
        $csrfToken = $this->auth->getCsrfToken();
        
        echo $this->renderLayout('auth.register', [
            'errors' => $errors,
            'csrf_token' => $csrfToken,
            'page_title' => 'Register'
        ]);
    }

    public function logout()
    {
        $this->auth->logout();
        $this->redirect('/');
    }

    public function forgotPassword()
    {
        $errors = [];
        $success = false;
        
        if ($_SERVER['REQUEST_METHOD'] === 'POST') {
            $this->validateCsrf();
            
            $email = $this->sanitizeInput($_POST['email'] ?? '');
            
            $validationErrors = $this->validateInput($_POST, [
                'email' => 'required|email'
            ]);
            
            if (empty($validationErrors)) {
                try {
                    $this->auth->sendPasswordReset($email);
                    $success = true;
                    $this->logger->logInfo("Password reset sent", ['email' => $email]);
                } catch (Exception $e) {
                    $errors[] = $e->getMessage();
                    $this->logger->logError("Password reset failed", ['email' => $email, 'error' => $e->getMessage()]);
                }
            } else {
                $errors = array_merge($errors, array_flatten($validationErrors));
            }
        }
        
        $csrfToken = $this->auth->getCsrfToken();
        
        echo $this->renderLayout('auth.forgot-password', [
            'errors' => $errors,
            'success' => $success,
            'csrf_token' => $csrfToken,
            'page_title' => 'Forgot Password'
        ]);
    }

    public function resetPassword()
    {
        $token = $_GET['token'] ?? '';
        $email = $_GET['email'] ?? '';
        
        if (empty($token) || empty($email)) {
            $this->redirect('/login');
        }
        
        $errors = [];
        $success = false;
        
        if ($_SERVER['REQUEST_METHOD'] === 'POST') {
            $this->validateCsrf();
            
            $password = $_POST['password'] ?? '';
            $confirmPassword = $_POST['confirm_password'] ?? '';
            
            $validationErrors = $this->validateInput($_POST, [
                'password' => 'required|min:8',
                'confirm_password' => 'required'
            ]);
            
            if (empty($validationErrors)) {
                if ($password !== $confirmPassword) {
                    $errors[] = 'Passwords do not match.';
                } else {
                    try {
                        $this->auth->resetPassword($email, $token, $password);
                        $success = true;
                        $this->logger->logInfo("Password reset successful", ['email' => $email]);
                    } catch (Exception $e) {
                        $errors[] = $e->getMessage();
                        $this->logger->logError("Password reset failed", ['email' => $email, 'error' => $e->getMessage()]);
                    }
                }
            } else {
                $errors = array_merge($errors, array_flatten($validationErrors));
            }
        }
        
        $csrfToken = $this->auth->getCsrfToken();
        
        echo $this->renderLayout('auth.reset-password', [
            'errors' => $errors,
            'success' => $success,
            'csrf_token' => $csrfToken,
            'token' => $token,
            'email' => $email,
            'page_title' => 'Reset Password'
        ]);
    }
}

function array_flatten($array)
{
    $result = [];
    
    foreach ($array as $item) {
        if (is_array($item)) {
            $result = array_merge($result, array_flatten($item));
        } else {
            $result[] = $item;
        }
    }
    
    return $result;
}