<?php

class BaseController
{
    protected $db;
    protected $auth;
    protected $logger;
    protected $viewPath;

    public function __construct()
    {
        $this->db = Database::getInstance();
        $this->auth = Auth::getInstance();
        $this->logger = Logger::getInstance();
        $this->viewPath = dirname(__DIR__) . '/views';
    }

    protected function render($view, $data = [])
    {
        extract($data);
        
        $viewFile = $this->viewPath . '/' . str_replace('.', '/', $view) . '.php';
        
        if (!file_exists($viewFile)) {
            throw new Exception("View file not found: {$view}");
        }
        
        ob_start();
        include $viewFile;
        $content = ob_get_clean();
        
        return $content;
    }

    protected function renderLayout($view, $data = [], $layout = 'layouts.main')
    {
        $content = $this->render($view, $data);
        
        $layoutData = array_merge($data, ['content' => $content]);
        
        return $this->render($layout, $layoutData);
    }

    protected function jsonResponse($data, $statusCode = 200)
    {
        http_response_code($statusCode);
        header('Content-Type: application/json');
        echo json_encode($data);
        exit;
    }

    protected function redirect($url, $code = 302)
    {
        header("Location: {$url}", true, $code);
        exit;
    }

    protected function validateCsrf()
    {
        $token = $_POST['csrf_token'] ?? $_SERVER['HTTP_X_CSRF_TOKEN'] ?? '';
        
        if (!$this->auth->validateCsrfToken($token)) {
            $this->logger->logWarning("CSRF validation failed", [
                'user_id' => $this->auth->getUserId(),
                'token' => $token
            ]);
            
            if (isset($_SERVER['HTTP_X_REQUESTED_WITH']) && $_SERVER['HTTP_X_REQUESTED_WITH'] === 'XMLHttpRequest') {
                $this->jsonResponse(['error' => 'Invalid CSRF token'], 403);
            } else {
                die('Invalid CSRF token');
            }
        }
    }

    protected function sanitizeInput($input)
    {
        if (is_array($input)) {
            return array_map([$this, 'sanitizeInput'], $input);
        }
        
        return htmlspecialchars(strip_tags(trim($input)), ENT_QUOTES, 'UTF-8');
    }

    protected function validateInput($data, $rules)
    {
        $errors = [];
        
        foreach ($rules as $field => $rule) {
            $value = $data[$field] ?? null;
            $rulesList = explode('|', $rule);
            
            foreach ($rulesList as $ruleItem) {
                $ruleParts = explode(':', $ruleItem);
                $ruleName = $ruleParts[0];
                $ruleValue = $ruleParts[1] ?? null;
                
                switch ($ruleName) {
                    case 'required':
                        if (empty($value) && $value !== '0') {
                            $errors[$field][] = "The {$field} field is required.";
                        }
                        break;
                        
                    case 'email':
                        if (!empty($value) && !filter_var($value, FILTER_VALIDATE_EMAIL)) {
                            $errors[$field][] = "The {$field} must be a valid email address.";
                        }
                        break;
                        
                    case 'min':
                        if (!empty($value) && strlen($value) < (int)$ruleValue) {
                            $errors[$field][] = "The {$field} must be at least {$ruleValue} characters.";
                        }
                        break;
                        
                    case 'max':
                        if (!empty($value) && strlen($value) > (int)$ruleValue) {
                            $errors[$field][] = "The {$field} may not be greater than {$ruleValue} characters.";
                        }
                        break;
                        
                    case 'url':
                        if (!empty($value) && !filter_var($value, FILTER_VALIDATE_URL)) {
                            $errors[$field][] = "The {$field} must be a valid URL.";
                        }
                        break;
                        
                    case 'numeric':
                        if (!empty($value) && !is_numeric($value)) {
                            $errors[$field][] = "The {$field} must be a number.";
                        }
                        break;
                }
            }
        }
        
        return $errors;
    }

    protected function paginate($totalItems, $currentPage, $perPage, $baseUrl)
    {
        $totalPages = ceil($totalItems / $perPage);
        $currentPage = max(1, min($currentPage, $totalPages ?: 1));
        
        $pages = [];
        $startPage = max(1, $currentPage - 2);
        $endPage = min($totalPages, $currentPage + 2);
        
        for ($i = $startPage; $i <= $endPage; $i++) {
            $pages[] = [
                'number' => $i,
                'url' => $baseUrl . '?page=' . $i,
                'active' => $i === $currentPage
            ];
        }
        
        return [
            'current_page' => $currentPage,
            'total_pages' => $totalPages,
            'per_page' => $perPage,
            'total_items' => $totalItems,
            'pages' => $pages,
            'has_previous' => $currentPage > 1,
            'has_next' => $currentPage < $totalPages,
            'previous_url' => $currentPage > 1 ? $baseUrl . '?page=' . ($currentPage - 1) : null,
            'next_url' => $currentPage < $totalPages ? $baseUrl . '?page=' . ($currentPage + 1) : null
        ];
    }
}