<?php

class User
{
    private $db;
    private $logger;

    public function __construct()
    {
        $this->db = Database::getInstance();
        $this->logger = Logger::getInstance();
    }

    public function create($email, $password, $role = 'member', $isActive = true)
    {
        $this->logger->logInfo("Creating new user", ['email' => $email, 'role' => $role]);
        
        if ($this->emailExists($email)) {
            throw new Exception("Email address already exists.");
        }
        
        $passwordHash = password_hash($password, PASSWORD_BCRYPT, ['cost' => 12]);
        $verificationToken = bin2hex(random_bytes(32));
        
        $this->db->execute(
            "INSERT INTO users (email, password_hash, role, is_active, verification_token, created_at) VALUES (?, ?, ?, ?, ?, NOW())",
            [$email, $passwordHash, $role, $isActive ? 1 : 0, $verificationToken]
        );
        
        $userId = $this->db->lastInsertId();
        
        $this->logger->logInfo("User created successfully", ['user_id' => $userId, 'email' => $email]);
        
        return $userId;
    }

    public function update($userId, $data)
    {
        $this->logger->logInfo("Updating user", ['user_id' => $userId, 'data' => array_keys($data)]);
        
        $allowedFields = ['email', 'role', 'is_active'];
        $fields = [];
        $params = [];
        
        foreach ($data as $field => $value) {
            if (in_array($field, $allowedFields)) {
                $fields[] = "{$field} = ?";
                $params[] = $value;
            }
        }
        
        if (empty($fields)) {
            throw new Exception("No valid fields to update.");
        }
        
        $params[] = $userId;
        $sql = "UPDATE users SET " . implode(', ', $fields) . ", updated_at = NOW() WHERE id = ?";
        
        $this->db->execute($sql, $params);
        
        $this->logger->logInfo("User updated successfully", ['user_id' => $userId]);
        
        return true;
    }

    public function delete($userId)
    {
        $this->logger->logInfo("Deleting user", ['user_id' => $userId]);
        
        $this->db->execute(
            "DELETE FROM users WHERE id = ?",
            [$userId]
        );
        
        $this->logger->logInfo("User deleted successfully", ['user_id' => $userId]);
        
        return true;
    }

    public function getById($userId)
    {
        return $this->db->fetch(
            "SELECT id, email, role, is_active, email_verified_at, last_login_at, last_login_ip, created_at 
             FROM users 
             WHERE id = ?",
            [$userId]
        );
    }

    public function getByEmail($email)
    {
        return $this->db->fetch(
            "SELECT id, email, password_hash, role, is_active, email_verified_at, last_login_at, last_login_ip, created_at 
             FROM users 
             WHERE email = ?",
            [$email]
        );
    }

    public function getAll($limit = 100, $offset = 0)
    {
        return $this->db->fetchAll(
            "SELECT id, email, role, is_active, email_verified_at, last_login_at, created_at 
             FROM users 
             ORDER BY created_at DESC 
             LIMIT ? OFFSET ?",
            [$limit, $offset]
        );
    }

    public function search($query, $limit = 50)
    {
        $searchTerm = "%{$query}%";
        
        return $this->db->fetchAll(
            "SELECT id, email, role, is_active, email_verified_at, last_login_at, created_at 
             FROM users 
             WHERE email LIKE ? OR role LIKE ?
             ORDER BY created_at DESC 
             LIMIT ?",
            [$searchTerm, $searchTerm, $limit]
        );
    }

    public function updatePassword($userId, $newPassword)
    {
        $this->logger->logInfo("Updating user password", ['user_id' => $userId]);
        
        $passwordHash = password_hash($newPassword, PASSWORD_BCRYPT, ['cost' => 12]);
        
        $this->db->execute(
            "UPDATE users SET password_hash = ?, updated_at = NOW() WHERE id = ?",
            [$passwordHash, $userId]
        );
        
        $this->logger->logInfo("Password updated successfully", ['user_id' => $userId]);
        
        return true;
    }

    public function verifyEmail($userId)
    {
        $this->logger->logInfo("Verifying user email", ['user_id' => $userId]);
        
        $this->db->execute(
            "UPDATE users SET email_verified_at = NOW(), verification_token = NULL WHERE id = ?",
            [$userId]
        );
        
        $this->logger->logInfo("Email verified successfully", ['user_id' => $userId]);
        
        return true;
    }

    public function suspend($userId)
    {
        $this->logger->logInfo("Suspending user", ['user_id' => $userId]);
        
        return $this->update($userId, ['is_active' => 0]);
    }

    public function activate($userId)
    {
        $this->logger->logInfo("Activating user", ['user_id' => $userId]);
        
        return $this->update($userId, ['is_active' => 1]);
    }

    public function getStats()
    {
        return $this->db->fetch(
            "SELECT 
                COUNT(*) as total_users,
                SUM(CASE WHEN role = 'admin' THEN 1 ELSE 0 END) as admin_count,
                SUM(CASE WHEN role = 'member' THEN 1 ELSE 0 END) as member_count,
                SUM(CASE WHEN is_active = 1 THEN 1 ELSE 0 END) as active_users,
                SUM(CASE WHEN is_active = 0 THEN 1 ELSE 0 END) as suspended_users,
                COUNT(CASE WHEN DATE(created_at) = CURDATE() THEN 1 END) as users_today
             FROM users"
        );
    }

    private function emailExists($email)
    {
        $result = $this->db->fetch(
            "SELECT COUNT(*) as count FROM users WHERE email = ?",
            [$email]
        );
        
        return $result['count'] > 0;
    }
}