<?php

class Website
{
    private $db;
    private $logger;

    public function __construct()
    {
        $this->db = Database::getInstance();
        $this->logger = Logger::getInstance();
    }

    public function create($userId, $name, $url, $description = null, $industry = null, $targetKeywords = null)
    {
        $this->logger->logInfo("Creating website", ['user_id' => $userId, 'name' => $name, 'url' => $url]);
        
        $this->validateUrl($url);
        
        $this->db->execute(
            "INSERT INTO websites (user_id, name, url, description, industry, target_keywords, created_at) 
             VALUES (?, ?, ?, ?, ?, ?, NOW())",
            [$userId, $name, $url, $description, $industry, $targetKeywords]
        );
        
        $websiteId = $this->db->lastInsertId();
        
        $this->logger->logInfo("Website created successfully", ['website_id' => $websiteId, 'name' => $name]);
        
        return $websiteId;
    }

    public function update($websiteId, $data)
    {
        $this->logger->logInfo("Updating website", ['website_id' => $websiteId, 'data' => array_keys($data)]);
        
        $allowedFields = ['name', 'url', 'description', 'industry', 'target_keywords', 'is_active'];
        $fields = [];
        $params = [];
        
        foreach ($data as $field => $value) {
            if (in_array($field, $allowedFields)) {
                if ($field === 'url') {
                    $this->validateUrl($value);
                }
                $fields[] = "{$field} = ?";
                $params[] = $value;
            }
        }
        
        if (empty($fields)) {
            throw new Exception("No valid fields to update.");
        }
        
        $params[] = $websiteId;
        $sql = "UPDATE websites SET " . implode(', ', $fields) . ", updated_at = NOW() WHERE id = ?";
        
        $this->db->execute($sql, $params);
        
        $this->logger->logInfo("Website updated successfully", ['website_id' => $websiteId]);
        
        return true;
    }

    public function delete($websiteId)
    {
        $this->logger->logInfo("Deleting website", ['website_id' => $websiteId]);
        
        $this->db->execute(
            "DELETE FROM websites WHERE id = ?",
            [$websiteId]
        );
        
        $this->logger->logInfo("Website deleted successfully", ['website_id' => $websiteId]);
        
        return true;
    }

    public function getById($websiteId)
    {
        return $this->db->fetch(
            "SELECT w.*, 
                    (SELECT AVG(visibility_score) FROM analytics WHERE website_id = w.id) as avg_visibility_score,
                    (SELECT COUNT(*) FROM analytics WHERE website_id = w.id AND mention_detected = 1) as mention_count,
                    (SELECT COUNT(*) FROM analytics WHERE website_id = w.id AND recommendation_detected = 1) as recommendation_count
             FROM websites w 
             WHERE w.id = ?",
            [$websiteId]
        );
    }

    public function getByUserId($userId, $limit = 100, $offset = 0)
    {
        return $this->db->fetchAll(
            "SELECT w.*, 
                    (SELECT AVG(visibility_score) FROM analytics WHERE website_id = w.id) as avg_visibility_score,
                    (SELECT COUNT(*) FROM analytics WHERE website_id = w.id AND mention_detected = 1) as mention_count,
                    (SELECT COUNT(*) FROM analytics WHERE website_id = w.id AND recommendation_detected = 1) as recommendation_count
             FROM websites w 
             WHERE w.user_id = ? 
             ORDER BY w.created_at DESC 
             LIMIT ? OFFSET ?",
            [$userId, $limit, $offset]
        );
    }

    public function getAll($limit = 100, $offset = 0)
    {
        return $this->db->fetchAll(
            "SELECT w.*, u.email as user_email,
                    (SELECT AVG(visibility_score) FROM analytics WHERE website_id = w.id) as avg_visibility_score,
                    (SELECT COUNT(*) FROM analytics WHERE website_id = w.id AND mention_detected = 1) as mention_count,
                    (SELECT COUNT(*) FROM analytics WHERE website_id = w.id AND recommendation_detected = 1) as recommendation_count
             FROM websites w 
             JOIN users u ON w.user_id = u.id 
             ORDER BY w.created_at DESC 
             LIMIT ? OFFSET ?",
            [$limit, $offset]
        );
    }

    public function search($query, $limit = 50)
    {
        $searchTerm = "%{$query}%";
        
        return $this->db->fetchAll(
            "SELECT w.*, u.email as user_email,
                    (SELECT AVG(visibility_score) FROM analytics WHERE website_id = w.id) as avg_visibility_score
             FROM websites w 
             JOIN users u ON w.user_id = u.id 
             WHERE w.name LIKE ? OR w.url LIKE ? OR u.email LIKE ?
             ORDER BY w.created_at DESC 
             LIMIT ?",
            [$searchTerm, $searchTerm, $searchTerm, $limit]
        );
    }

    public function updateVisibilityScore($websiteId, $score)
    {
        $this->logger->logInfo("Updating visibility score", ['website_id' => $websiteId, 'score' => $score]);
        
        $this->db->execute(
            "UPDATE websites SET visibility_score = ?, updated_at = NOW() WHERE id = ?",
            [$score, $websiteId]
        );
        
        return true;
    }

    public function updateRecommendationScore($websiteId, $score)
    {
        $this->logger->logInfo("Updating recommendation score", ['website_id' => $websiteId, 'score' => $score]);
        
        $this->db->execute(
            "UPDATE websites SET recommendation_score = ?, updated_at = NOW() WHERE id = ?",
            [$score, $websiteId]
        );
        
        return true;
    }

    public function updateLastScan($websiteId)
    {
        $this->db->execute(
            "UPDATE websites SET last_scan_at = NOW() WHERE id = ?",
            [$websiteId]
        );
        
        return true;
    }

    public function getStatsByUser($userId)
    {
        return $this->db->fetch(
            "SELECT 
                COUNT(*) as total_websites,
                SUM(CASE WHEN is_active = 1 THEN 1 ELSE 0 END) as active_websites,
                SUM(CASE WHEN is_active = 0 THEN 1 ELSE 0 END) as inactive_websites,
                AVG(visibility_score) as avg_visibility_score,
                MAX(last_scan_at) as last_scan_at
             FROM websites 
             WHERE user_id = ?",
            [$userId]
        );
    }

    public function getGlobalStats()
    {
        return $this->db->fetch(
            "SELECT 
                COUNT(*) as total_websites,
                AVG(visibility_score) as avg_visibility_score,
                AVG(recommendation_score) as avg_recommendation_score,
                COUNT(CASE WHEN last_scan_at IS NOT NULL THEN 1 END) as scanned_websites,
                MAX(last_scan_at) as last_global_scan
             FROM websites 
             WHERE is_active = 1"
        );
    }

    public function getWebsitesNeedingScan($hours = 24)
    {
        return $this->db->fetchAll(
            "SELECT id, user_id, name, url, industry, target_keywords, last_scan_at
             FROM websites 
             WHERE is_active = 1 
             AND (last_scan_at IS NULL OR last_scan_at < DATE_SUB(NOW(), INTERVAL ? HOUR))
             ORDER BY last_scan_at ASC",
            [$hours]
        );
    }

    private function validateUrl($url)
    {
        if (!filter_var($url, FILTER_VALIDATE_URL)) {
            throw new Exception("Invalid URL format.");
        }
        
        if (!preg_match('/^https?:\/\//', $url)) {
            throw new Exception("URL must start with http:// or https://");
        }
        
        return true;
    }
}