<?php

class RateLimiterService
{
    private $db;
    private $logger;

    public function __construct()
    {
        $this->db = Database::getInstance();
        $this->logger = Logger::getInstance();
    }

    public function checkRateLimit($userId, $modelId, $action)
    {
        $limits = $this->getRateLimits($modelId, $action);
        
        foreach ($limits as $limit) {
            $window = $limit['window_minutes'] * 60;
            $maxRequests = $limit['max_requests'];
            
            $count = $this->db->fetch(
                "SELECT COUNT(*) as count 
                 FROM rate_limit_logs 
                 WHERE user_id = ? AND model_id = ? AND action = ? 
                 AND created_at > DATE_SUB(NOW(), INTERVAL ? SECOND)",
                [$userId, $modelId, $action, $window]
            );
            
            if ($count['count'] >= $maxRequests) {
                $this->logger->logWarning("Rate limit exceeded", [
                    'user_id' => $userId,
                    'model_id' => $modelId,
                    'action' => $action,
                    'limit' => $maxRequests,
                    'window' => $limit['window_minutes'],
                    'current_count' => $count['count']
                ]);
                
                return false;
            }
        }
        
        $this->db->execute(
            "INSERT INTO rate_limit_logs (user_id, model_id, action, created_at) VALUES (?, ?, ?, NOW())",
            [$userId, $modelId, $action]
        );
        
        return true;
    }

    public function checkUserRateLimit($userId, $action, $windowMinutes = 60, $maxRequests = 100)
    {
        $window = $windowMinutes * 60;
        
        $count = $this->db->fetch(
            "SELECT COUNT(*) as count 
             FROM rate_limit_logs 
             WHERE user_id = ? AND action = ? 
             AND created_at > DATE_SUB(NOW(), INTERVAL ? SECOND)",
            [$userId, $action, $window]
        );
        
        if ($count['count'] >= $maxRequests) {
            $this->logger->logWarning("User rate limit exceeded", [
                'user_id' => $userId,
                'action' => $action,
                'limit' => $maxRequests,
                'window' => $windowMinutes,
                'current_count' => $count['count']
            ]);
            
            return false;
        }
        
        return true;
    }

    private function getRateLimits($modelId, $action)
    {
        $modelLimits = $this->db->fetchAll(
            "SELECT window_minutes, max_requests 
             FROM rate_limits 
             WHERE model_id = ? AND action = ? AND is_active = 1",
            [$modelId, $action]
        );
        
        if (!empty($modelLimits)) {
            return $modelLimits;
        }
        
        $model = $this->db->fetch(
            "SELECT max_requests_per_minute, max_requests_per_hour FROM ai_models WHERE id = ?",
            [$modelId]
        );
        
        if ($model) {
            return [
                ['window_minutes' => 1, 'max_requests' => $model['max_requests_per_minute']],
                ['window_minutes' => 60, 'max_requests' => $model['max_requests_per_hour']]
            ];
        }
        
        return [
            ['window_minutes' => 1, 'max_requests' => 60],
            ['window_minutes' => 60, 'max_requests' => 1000]
        ];
    }

    public function getUserStats($userId, $hours = 24)
    {
        return $this->db->fetch(
            "SELECT 
                COUNT(*) as total_requests,
                COUNT(DISTINCT model_id) as models_used,
                COUNT(DISTINCT action) as actions_performed,
                GROUP_CONCAT(DISTINCT action) as action_list
             FROM rate_limit_logs 
             WHERE user_id = ? AND created_at > DATE_SUB(NOW(), INTERVAL ? HOUR)",
            [$userId, $hours]
        );
    }

    public function clearOldLogs($olderThanDays = 30)
    {
        $this->db->execute(
            "DELETE FROM rate_limit_logs WHERE created_at < DATE_SUB(NOW(), INTERVAL ? DAY)",
            [$olderThanDays]
        );
        
        $this->logger->logInfo("Old rate limit logs cleared", ['older_than_days' => $olderThanDays]);
    }

    public function resetUserLimits($userId, $modelId = null, $action = null)
    {
        $sql = "DELETE FROM rate_limit_logs WHERE user_id = ?";
        $params = [$userId];
        
        if ($modelId !== null) {
            $sql .= " AND model_id = ?";
            $params[] = $modelId;
        }
        
        if ($action !== null) {
            $sql .= " AND action = ?";
            $params[] = $action;
        }
        
        $this->db->execute($sql, $params);
        
        $this->logger->logInfo("User rate limits reset", [
            'user_id' => $userId,
            'model_id' => $modelId,
            'action' => $action
        ]);
        
        return true;
    }
}