<?php

class Auth
{
    private static $instance = null;
    private $db;
    private $logger;
    private $sessionStarted = false;

    private function __construct()
    {
        $this->db = Database::getInstance();
        $this->logger = Logger::getInstance();
        
        $this->initSession();
    }

    public static function getInstance()
    {
        if (self::$instance === null) {
            self::$instance = new self();
        }
        return self::$instance;
    }

    private function initSession()
    {
        if (session_status() === PHP_SESSION_NONE) {
            session_set_cookie_params([
                'lifetime' => 86400,
                'path' => '/',
                'domain' => '',
                'secure' => isset($_SERVER['HTTPS']) && $_SERVER['HTTPS'] === 'on',
                'httponly' => true,
                'samesite' => 'Strict'
            ]);
            
            session_start();
            $this->sessionStarted = true;
            
            $this->regenerateSession();
        }
    }

    private function regenerateSession()
    {
        if (!isset($_SESSION['last_regeneration'])) {
            $_SESSION['last_regeneration'] = time();
        }
        
        if (time() - $_SESSION['last_regeneration'] > 1800) {
            session_regenerate_id(true);
            $_SESSION['last_regeneration'] = time();
        }
    }

    public function login($email, $password)
    {
        $this->logger->logInfo("Login attempt", ['email' => $email]);
        
        if ($this->isBruteForceAttempt($email)) {
            $this->logger->logWarning("Brute force login attempt detected", ['email' => $email]);
            throw new Exception("Too many login attempts. Please try again later.");
        }
        
        $user = $this->db->fetch(
            "SELECT id, email, password_hash, role, is_active, email_verified_at FROM users WHERE email = ? LIMIT 1",
            [$email]
        );
        
        if (!$user) {
            $this->logFailedAttempt($email);
            $this->logger->logWarning("Login failed - user not found", ['email' => $email]);
            throw new Exception("Invalid email or password.");
        }
        
        if (!$user['is_active']) {
            $this->logger->logWarning("Login failed - inactive account", ['email' => $email, 'user_id' => $user['id']]);
            throw new Exception("Account is deactivated. Please contact support.");
        }
        
        if (!password_verify($password, $user['password_hash'])) {
            $this->logFailedAttempt($email);
            $this->logger->logWarning("Login failed - invalid password", ['email' => $email, 'user_id' => $user['id']]);
            throw new Exception("Invalid email or password.");
        }
        
        $this->clearFailedAttempts($email);
        $this->createSession($user);
        
        $this->logger->logInfo("User logged in successfully", ['user_id' => $user['id'], 'email' => $email]);
        
        return true;
    }

    private function createSession($user)
    {
        $_SESSION['user_id'] = $user['id'];
        $_SESSION['user_email'] = $user['email'];
        $_SESSION['user_role'] = $user['role'];
        $_SESSION['logged_in'] = true;
        $_SESSION['login_time'] = time();
        $_SESSION['csrf_token'] = $this->generateCsrfToken();
        
        $this->db->execute(
            "UPDATE users SET last_login_at = NOW(), last_login_ip = ? WHERE id = ?",
            [$_SERVER['REMOTE_ADDR'] ?? null, $user['id']]
        );
    }

    public function logout()
    {
        if (isset($_SESSION['user_id'])) {
            $this->logger->logInfo("User logged out", ['user_id' => $_SESSION['user_id']]);
        }
        
        $_SESSION = [];
        
        if (ini_get("session.use_cookies")) {
            $params = session_get_cookie_params();
            setcookie(
                session_name(),
                '',
                time() - 42000,
                $params["path"],
                $params["domain"],
                $params["secure"],
                $params["httponly"]
            );
        }
        
        session_destroy();
    }

    public function isLoggedIn()
    {
        return isset($_SESSION['logged_in']) && $_SESSION['logged_in'] === true;
    }

    public function getUserId()
    {
        return $_SESSION['user_id'] ?? null;
    }

    public function getUserEmail()
    {
        return $_SESSION['user_email'] ?? null;
    }

    public function getUserRole()
    {
        return $_SESSION['user_role'] ?? null;
    }

    public function isAdmin()
    {
        return $this->getUserRole() === 'admin';
    }

    public function requireLogin()
    {
        if (!$this->isLoggedIn()) {
            $this->logger->logWarning("Unauthorized access attempt", ['url' => $_SERVER['REQUEST_URI'] ?? null]);
            
            if (isset($_SERVER['HTTP_X_REQUESTED_WITH']) && $_SERVER['HTTP_X_REQUESTED_WITH'] === 'XMLHttpRequest') {
                http_response_code(401);
                echo json_encode(['error' => 'Unauthorized']);
                exit;
            } else {
                header('Location: /login');
                exit;
            }
        }
    }

    public function requireAdmin()
    {
        $this->requireLogin();
        
        if (!$this->isAdmin()) {
            $this->logger->logWarning("Admin access denied", ['user_id' => $this->getUserId(), 'url' => $_SERVER['REQUEST_URI'] ?? null]);
            
            if (isset($_SERVER['HTTP_X_REQUESTED_WITH']) && $_SERVER['HTTP_X_REQUESTED_WITH'] === 'XMLHttpRequest') {
                http_response_code(403);
                echo json_encode(['error' => 'Forbidden']);
                exit;
            } else {
                header('Location: /dashboard');
                exit;
            }
        }
    }

    public function generateCsrfToken()
    {
        return bin2hex(random_bytes(32));
    }

    public function getCsrfToken()
    {
        if (!isset($_SESSION['csrf_token'])) {
            $_SESSION['csrf_token'] = $this->generateCsrfToken();
        }
        
        return $_SESSION['csrf_token'];
    }

    public function validateCsrfToken($token)
    {
        if (!isset($_SESSION['csrf_token'])) {
            return false;
        }
        
        return hash_equals($_SESSION['csrf_token'], $token);
    }

    public function register($email, $password, $role = 'member')
    {
        $this->logger->logInfo("Registration attempt", ['email' => $email]);
        
        if ($this->emailExists($email)) {
            $this->logger->logWarning("Registration failed - email already exists", ['email' => $email]);
            throw new Exception("Email address is already registered.");
        }
        
        $passwordHash = password_hash($password, PASSWORD_BCRYPT, ['cost' => 12]);
        $verificationToken = bin2hex(random_bytes(32));
        
        $this->db->execute(
            "INSERT INTO users (email, password_hash, role, verification_token, created_at) VALUES (?, ?, ?, ?, NOW())",
            [$email, $passwordHash, $role, $verificationToken]
        );
        
        $userId = $this->db->lastInsertId();
        
        $this->logger->logInfo("User registered successfully", ['user_id' => $userId, 'email' => $email]);
        
        return $userId;
    }

    public function emailExists($email)
    {
        $result = $this->db->fetch(
            "SELECT COUNT(*) as count FROM users WHERE email = ?",
            [$email]
        );
        
        return $result['count'] > 0;
    }

    private function isBruteForceAttempt($email)
    {
        $timeWindow = time() - 900;
        
        $attempts = $this->db->fetch(
            "SELECT COUNT(*) as count FROM login_attempts WHERE email = ? AND attempt_time > FROM_UNIXTIME(?)",
            [$email, $timeWindow]
        );
        
        return $attempts['count'] >= 5;
    }

    private function logFailedAttempt($email)
    {
        $this->db->execute(
            "INSERT INTO login_attempts (email, attempt_time, ip_address) VALUES (?, NOW(), ?)",
            [$email, $_SERVER['REMOTE_ADDR'] ?? null]
        );
    }

    private function clearFailedAttempts($email)
    {
        $this->db->execute(
            "DELETE FROM login_attempts WHERE email = ?",
            [$email]
        );
    }

    public function sendPasswordReset($email)
    {
        $user = $this->db->fetch(
            "SELECT id, email FROM users WHERE email = ? AND is_active = 1 LIMIT 1",
            [$email]
        );
        
        if (!$user) {
            $this->logger->logWarning("Password reset requested for non-existent email", ['email' => $email]);
            return true;
        }
        
        $token = bin2hex(random_bytes(32));
        $expires = time() + 3600;
        
        $this->db->execute(
            "INSERT INTO password_resets (user_id, token, expires_at) VALUES (?, ?, FROM_UNIXTIME(?))",
            [$user['id'], password_hash($token, PASSWORD_BCRYPT), $expires]
        );
        
        $resetUrl = BASE_URL . "/reset-password?token=" . urlencode($token) . "&email=" . urlencode($email);
        
        $subject = "Password Reset Request";
        $message = "Click the link below to reset your password:\n\n" . $resetUrl . "\n\nThis link expires in 1 hour.";
        
        $headers = [
            'From' => 'noreply@' . $_SERVER['HTTP_HOST'],
            'Content-Type' => 'text/plain; charset=UTF-8',
            'X-Mailer' => 'PHP/' . phpversion()
        ];
        
        $headersString = '';
        foreach ($headers as $key => $value) {
            $headersString .= $key . ': ' . $value . "\r\n";
        }
        
        mail($email, $subject, $message, $headersString);
        
        $this->logger->logInfo("Password reset email sent", ['user_id' => $user['id'], 'email' => $email]);
        
        return true;
    }

    public function resetPassword($email, $token, $newPassword)
    {
        $user = $this->db->fetch(
            "SELECT id FROM users WHERE email = ? AND is_active = 1 LIMIT 1",
            [$email]
        );
        
        if (!$user) {
            throw new Exception("Invalid reset request.");
        }
        
        $reset = $this->db->fetch(
            "SELECT token, expires_at FROM password_resets WHERE user_id = ? ORDER BY created_at DESC LIMIT 1",
            [$user['id']]
        );
        
        if (!$reset || !password_verify($token, $reset['token'])) {
            throw new Exception("Invalid or expired reset token.");
        }
        
        if (strtotime($reset['expires_at']) < time()) {
            throw new Exception("Reset token has expired.");
        }
        
        $passwordHash = password_hash($newPassword, PASSWORD_BCRYPT, ['cost' => 12]);
        
        $this->db->execute(
            "UPDATE users SET password_hash = ? WHERE id = ?",
            [$passwordHash, $user['id']]
        );
        
        $this->db->execute(
            "DELETE FROM password_resets WHERE user_id = ?",
            [$user['id']]
        );
        
        $this->logger->logInfo("Password reset successful", ['user_id' => $user['id'], 'email' => $email]);
        
        return true;
    }
}