<?php

class Database
{
    private static $instance = null;
    private $connection;
    private $host;
    private $database;
    private $username;
    private $password;
    private $logger;

    private function __construct()
    {
        $this->logger = Logger::getInstance();
        
        if (defined('DB_HOST')) {
            $this->host = DB_HOST;
            $this->database = DB_NAME;
            $this->username = DB_USER;
            $this->password = DB_PASS;
            $this->connect();
        }
    }

    public static function getInstance()
    {
        if (self::$instance === null) {
            self::$instance = new self();
        }
        return self::$instance;
    }

    public function connect()
    {
        try {
            $dsn = "mysql:host={$this->host};dbname={$this->database};charset=utf8mb4";
            $options = [
                PDO::ATTR_ERRMODE => PDO::ERRMODE_EXCEPTION,
                PDO::ATTR_DEFAULT_FETCH_MODE => PDO::FETCH_ASSOC,
                PDO::ATTR_EMULATE_PREPARES => false,
                PDO::MYSQL_ATTR_INIT_COMMAND => "SET NAMES utf8mb4 COLLATE utf8mb4_unicode_ci"
            ];
            
            $this->connection = new PDO($dsn, $this->username, $this->password, $options);
            $this->logger->logInfo("Database connection established successfully");
            
            return $this->connection;
        } catch (PDOException $e) {
            $this->logger->logError("Database connection failed: " . $e->getMessage(), [
                'error_code' => 'DB_CONNECTION_FAILED',
                'host' => $this->host,
                'database' => $this->database
            ]);
            throw new Exception("Database connection failed. Please check your configuration.");
        }
    }

    public function getConnection()
    {
        if ($this->connection === null) {
            $this->connect();
        }
        
        try {
            $this->connection->query("SELECT 1");
        } catch (PDOException $e) {
            $this->connect();
        }
        
        return $this->connection;
    }

    public function query($sql, $params = [])
    {
        try {
            $stmt = $this->getConnection()->prepare($sql);
            $result = $stmt->execute($params);
            
            $this->logger->logDebug("Query executed successfully", [
                'sql' => $sql,
                'params' => $params
            ]);
            
            return $stmt;
        } catch (PDOException $e) {
            $this->logger->logError("Query execution failed: " . $e->getMessage(), [
                'error_code' => 'QUERY_FAILED',
                'sql' => $sql,
                'params' => $params
            ]);
            throw new Exception("Database query failed.");
        }
    }

    public function fetch($sql, $params = [])
    {
        $stmt = $this->query($sql, $params);
        return $stmt->fetch();
    }

    public function fetchAll($sql, $params = [])
    {
        $stmt = $this->query($sql, $params);
        return $stmt->fetchAll();
    }

    public function execute($sql, $params = [])
    {
        $stmt = $this->query($sql, $params);
        return $stmt->rowCount();
    }

    public function lastInsertId()
    {
        return $this->getConnection()->lastInsertId();
    }

    public function beginTransaction()
    {
        return $this->getConnection()->beginTransaction();
    }

    public function commit()
    {
        return $this->getConnection()->commit();
    }

    public function rollBack()
    {
        return $this->getConnection()->rollBack();
    }

    public function createTables($sql)
    {
        try {
            $this->getConnection()->exec($sql);
            $this->logger->logInfo("Tables created successfully");
            return true;
        } catch (PDOException $e) {
            $this->logger->logError("Table creation failed: " . $e->getMessage(), [
                'error_code' => 'TABLE_CREATE_FAILED'
            ]);
            return false;
        }
    }
}