<?php

class Logger
{
    private static $instance = null;
    private $logPath;
    private $debugMode;
    private $logLevel;
    
    const LEVEL_EMERGENCY = 0;
    const LEVEL_ALERT = 1;
    const LEVEL_CRITICAL = 2;
    const LEVEL_ERROR = 3;
    const LEVEL_WARNING = 4;
    const LEVEL_NOTICE = 5;
    const LEVEL_INFO = 6;
    const LEVEL_DEBUG = 7;
    
    private $levelNames = [
        self::LEVEL_EMERGENCY => 'EMERGENCY',
        self::LEVEL_ALERT => 'ALERT',
        self::LEVEL_CRITICAL => 'CRITICAL',
        self::LEVEL_ERROR => 'ERROR',
        self::LEVEL_WARNING => 'WARNING',
        self::LEVEL_NOTICE => 'NOTICE',
        self::LEVEL_INFO => 'INFO',
        self::LEVEL_DEBUG => 'DEBUG'
    ];

    private function __construct()
    {
        $this->logPath = dirname(__DIR__) . '/logs';
        $this->ensureLogDirectory();
        
        $this->debugMode = defined('DEBUG_MODE') ? DEBUG_MODE : false;
        $this->logLevel = defined('LOG_LEVEL') ? LOG_LEVEL : self::LEVEL_INFO;
    }

    public static function getInstance()
    {
        if (self::$instance === null) {
            self::$instance = new self();
        }
        return self::$instance;
    }

    private function ensureLogDirectory()
    {
        if (!is_dir($this->logPath)) {
            mkdir($this->logPath, 0755, true);
        }
        
        if (!is_writable($this->logPath)) {
            chmod($this->logPath, 0755);
        }
    }

    private function writeLog($level, $message, $context = [])
    {
        if ($level > $this->logLevel) {
            return;
        }

        $timestamp = date('Y-m-d H:i:s');
        $levelName = $this->levelNames[$level];
        $fileName = date('Y-m-d') . '.log';
        $filePath = $this->logPath . '/' . $fileName;

        $logEntry = [
            'timestamp' => $timestamp,
            'level' => $levelName,
            'message' => $message,
            'context' => $context
        ];

        $logLine = json_encode($logEntry, JSON_UNESCAPED_SLASHES | JSON_UNESCAPED_UNICODE) . PHP_EOL;

        try {
            file_put_contents($filePath, $logLine, FILE_APPEND | LOCK_EX);
        } catch (Exception $e) {
            error_log("Failed to write to log file: " . $e->getMessage());
        }
    }

    public function logDebug($message, $context = [])
    {
        if ($this->debugMode) {
            $this->writeLog(self::LEVEL_DEBUG, $message, $context);
        }
    }

    public function logInfo($message, $context = [])
    {
        $this->writeLog(self::LEVEL_INFO, $message, $context);
    }

    public function logNotice($message, $context = [])
    {
        $this->writeLog(self::LEVEL_NOTICE, $message, $context);
    }

    public function logWarning($message, $context = [])
    {
        $this->writeLog(self::LEVEL_WARNING, $message, $context);
    }

    public function logError($message, $context = [])
    {
        $this->writeLog(self::LEVEL_ERROR, $message, $context);
    }

    public function logCritical($message, $context = [])
    {
        $this->writeLog(self::LEVEL_CRITICAL, $message, $context);
    }

    public function logAlert($message, $context = [])
    {
        $this->writeLog(self::LEVEL_ALERT, $message, $context);
    }

    public function logEmergency($message, $context = [])
    {
        $this->writeLog(self::LEVEL_EMERGENCY, $message, $context);
    }

    public function getLogs($date = null, $level = null, $limit = 1000)
    {
        if ($date === null) {
            $date = date('Y-m-d');
        }

        $fileName = $date . '.log';
        $filePath = $this->logPath . '/' . $fileName;

        if (!file_exists($filePath)) {
            return [];
        }

        $logs = [];
        $lines = file($filePath, FILE_IGNORE_NEW_LINES | FILE_SKIP_EMPTY_LINES);
        
        foreach (array_reverse($lines) as $line) {
            if (count($logs) >= $limit) {
                break;
            }

            $logEntry = json_decode($line, true);
            if ($logEntry === null) {
                continue;
            }

            if ($level !== null && $logEntry['level'] !== $level) {
                continue;
            }

            $logs[] = $logEntry;
        }

        return $logs;
    }

    public function clearOldLogs($daysToKeep = 30)
    {
        $cutoffTime = time() - ($daysToKeep * 24 * 60 * 60);
        
        foreach (glob($this->logPath . '/*.log') as $file) {
            if (filemtime($file) < $cutoffTime) {
                unlink($file);
            }
        }
        
        $this->logInfo("Old logs cleared", ['days_to_keep' => $daysToKeep]);
    }
}