<?php
/**
 * AI Visibility Tracker - Public Entry Point
 */

// Start session
session_start();

// Define base path
define('BASE_PATH', dirname(__DIR__));

// Check if application is installed
if (!file_exists(BASE_PATH . '/config.php')) {
    if (file_exists(BASE_PATH . '/install/index.php')) {
        header('Location: /install/');
        exit;
    } else {
        die('Application is not installed and installer is missing.');
    }
}

// Load configuration
require_once BASE_PATH . '/config.php';

// Load core classes
require_once BASE_PATH . '/core/Logger.php';
require_once BASE_PATH . '/core/Database.php';
require_once BASE_PATH . '/core/Auth.php';
require_once BASE_PATH . '/core/Queue.php';

// Load models
require_once BASE_PATH . '/app/models/User.php';
require_once BASE_PATH . '/app/models/Website.php';

// Load services
require_once BASE_PATH . '/app/services/AIService.php';
require_once BASE_PATH . '/app/services/RateLimiterService.php';

// Load controllers
require_once BASE_PATH . '/app/controllers/BaseController.php';
require_once BASE_PATH . '/app/controllers/HomeController.php';
require_once BASE_PATH . '/app/controllers/AuthController.php';
require_once BASE_PATH . '/app/controllers/DashboardController.php';
require_once BASE_PATH . '/app/controllers/AdminController.php';

// Initialize logger
$logger = Logger::getInstance();

// Initialize auth
$auth = Auth::getInstance();

// Set up error handler
set_error_handler(function($errno, $errstr, $errfile, $errline) use ($logger) {
    $logger->logError($errstr, [
        'error_code' => 'PHP_ERROR_' . $errno,
        'file' => $errfile,
        'line' => $errline
    ]);
    
    if (DEBUG_MODE) {
        return false; // Let PHP handle the error
    }
    
    return true;
});

// Set up exception handler
set_exception_handler(function($e) use ($logger) {
    $logger->logError($e->getMessage(), [
        'error_code' => 'EXCEPTION',
        'file' => $e->getFile(),
        'line' => $e->getLine(),
        'trace' => $e->getTraceAsString()
    ]);
    
    http_response_code(500);
    
    if (DEBUG_MODE) {
        echo '<h1>Exception</h1>';
        echo '<p>' . htmlspecialchars($e->getMessage()) . '</p>';
        echo '<pre>' . htmlspecialchars($e->getTraceAsString()) . '</pre>';
    } else {
        echo '<h1>500 Internal Server Error</h1>';
        echo '<p>Something went wrong. Please try again later.</p>';
    }
});

// Simple router
try {
    $request = parse_url($_SERVER['REQUEST_URI']);
    $path = $request['path'];
    $path = rtrim($path, '/');
    
    // Remove public prefix if present
    if (strpos($path, '/public') === 0) {
        $path = substr($path, 7);
    }
    
    $logger->logDebug('Routing request', ['path' => $path]);
    
    switch ($path) {
        // Home page
        case '':
        case '/':
            $controller = new HomeController();
            $controller->index();
            break;
            
        // Static pages
        case '/features':
            $controller = new HomeController();
            $controller->features();
            break;
            
        case '/pricing':
            $controller = new HomeController();
            $controller->pricing();
            break;
            
        case '/about':
            $controller = new HomeController();
            $controller->about();
            break;
            
        case '/contact':
            $controller = new HomeController();
            $controller->contact();
            break;
            
        case '/terms':
            $controller = new HomeController();
            $controller->terms();
            break;
            
        case '/privacy':
            $controller = new HomeController();
            $controller->privacy();
            break;
            
        // Authentication
        case '/login':
            $controller = new AuthController();
            $controller->login();
            break;
            
        case '/register':
            $controller = new AuthController();
            $controller->register();
            break;
            
        case '/logout':
            $controller = new AuthController();
            $controller->logout();
            break;
            
        case '/forgot-password':
            $controller = new AuthController();
            $controller->forgotPassword();
            break;
            
        case '/reset-password':
            $controller = new AuthController();
            $controller->resetPassword();
            break;
            
        // Dashboard
        case '/dashboard':
            $controller = new DashboardController();
            $controller->index();
            break;
            
        case '/dashboard/websites':
            $controller = new DashboardController();
            $controller->websites();
            break;
            
        case '/dashboard/websites/add':
            $controller = new DashboardController();
            $controller->addWebsite();
            break;
            
        // API endpoints
        case '/api/job-status':
            handleAPIRequest();
            break;
            
        default:
            // Handle dynamic routes
            if (preg_match('#^/dashboard/websites/([0-9]+)$#', $path, $matches)) {
                $controller = new DashboardController();
                $controller->viewWebsite((int)$matches[1]);
            } elseif (preg_match('#^/dashboard/websites/([0-9]+)/run-test$#', $path, $matches)) {
                $controller = new DashboardController();
                $controller->runAITest((int)$matches[1]);
            } elseif (preg_match('#^/dashboard/websites/([0-9]+)/delete$#', $path, $matches)) {
                $controller = new DashboardController();
                $controller->deleteWebsite((int)$matches[1]);
            } elseif (preg_match('#^/dashboard/analytics$#', $path)) {
                $controller = new DashboardController();
                $controller->analytics();
            } elseif (preg_match('#^/dashboard/settings$#', $path)) {
                $controller = new DashboardController();
                $controller->settings();
            } elseif (preg_match('#^/admin$#', $path)) {
                $controller = new AdminController();
                $controller->index();
            } elseif (preg_match('#^/admin/users$#', $path)) {
                $controller = new AdminController();
                $controller->users();
            } elseif (preg_match('#^/admin/users/([0-9]+)$#', $path, $matches)) {
                $controller = new AdminController();
                $controller->editUser((int)$matches[1]);
            } elseif (preg_match('#^/admin/users/([0-9]+)/reset-password$#', $path, $matches)) {
                $controller = new AdminController();
                $controller->resetUserPassword((int)$matches[1]);
            } elseif (preg_match('#^/admin/websites$#', $path)) {
                $controller = new AdminController();
                $controller->websites();
            } elseif (preg_match('#^/admin/websites/([0-9]+)$#', $path, $matches)) {
                $controller = new AdminController();
                $controller->viewWebsite((int)$matches[1]);
            } elseif (preg_match('#^/admin/websites/([0-9]+)/rescan$#', $path, $matches)) {
                $controller = new AdminController();
                $controller->forceRescan((int)$matches[1]);
            } elseif (preg_match('#^/admin/models$#', $path)) {
                $controller = new AdminController();
                $controller->models();
            } elseif (preg_match('#^/admin/models/([0-9]+)$#', $path, $matches)) {
                $controller = new AdminController();
                $controller->editModel((int)$matches[1]);
            } elseif (preg_match('#^/admin/prompts$#', $path)) {
                $controller = new AdminController();
                $controller->prompts();
            } elseif (preg_match('#^/admin/prompts/([0-9]+)$#', $path, $matches)) {
                $controller = new AdminController();
                $controller->editPrompt((int)$matches[1]);
            } elseif (preg_match('#^/admin/settings$#', $path)) {
                $controller = new AdminController();
                $controller->settings();
            } elseif (preg_match('#^/admin/logs$#', $path)) {
                $controller = new AdminController();
                $controller->logs();
            } else {
                // Try to serve static files
                $filePath = BASE_PATH . '/public' . $path;
                
                if (file_exists($filePath) && is_file($filePath)) {
                    serveStaticFile($filePath);
                } else {
                    http_response_code(404);
                    echo '<h1>404 Not Found</h1>';
                    echo '<p>The page you are looking for does not exist.</p>';
                }
            }
    }
    
} catch (Exception $e) {
    $logger->logError('Router error', ['error' => $e->getMessage(), 'path' => $path ?? 'unknown']);
    
    http_response_code(500);
    echo '<h1>500 Internal Server Error</h1>';
    echo '<p>Failed to process your request.</p>';
}

function handleAPIRequest()
{
    header('Content-Type: application/json');
    
    $method = $_SERVER['REQUEST_METHOD'];
    $path = parse_url($_SERVER['REQUEST_URI'], PHP_URL_PATH);
    
    if (preg_match('#/api/job-status/([0-9]+)#', $path, $matches)) {
        $jobId = (int)$matches[1];
        
        $db = Database::getInstance();
        $job = $db->fetch(
            "SELECT status, result, error_message FROM job_queue WHERE id = ?",
            [$jobId]
        );
        
        if ($job) {
            echo json_encode([
                'status' => $job['status'],
                'result' => $job['result'] ? json_decode($job['result'], true) : null,
                'error' => $job['error_message']
            ]);
        } else {
            http_response_code(404);
            echo json_encode(['error' => 'Job not found']);
        }
    } else {
        http_response_code(404);
        echo json_encode(['error' => 'API endpoint not found']);
    }
}

function serveStaticFile($filePath)
{
    $mimeTypes = [
        'css' => 'text/css',
        'js' => 'application/javascript',
        'png' => 'image/png',
        'jpg' => 'image/jpeg',
        'jpeg' => 'image/jpeg',
        'gif' => 'image/gif',
        'svg' => 'image/svg+xml',
        'ico' => 'image/x-icon',
        'pdf' => 'application/pdf',
        'woff' => 'font/woff',
        'woff2' => 'font/woff2',
        'ttf' => 'font/ttf'
    ];
    
    $extension = pathinfo($filePath, PATHINFO_EXTENSION);
    $contentType = $mimeTypes[$extension] ?? 'application/octet-stream';
    
    header('Content-Type: ' . $contentType);
    header('Content-Length: ' . filesize($filePath));
    
    // Set cache headers for static files
    if (!in_array($extension, ['html', 'php'])) {
        header('Cache-Control: public, max-age=31536000');
        header('Expires: ' . gmdate('D, d M Y H:i:s', time() + 31536000) . ' GMT');
    }
    
    readfile($filePath);
    exit;
}