// AI Visibility Tracker - Main JavaScript

(function() {
    'use strict';

    // Mobile navigation toggle
    const navToggle = document.querySelector('.nav-toggle');
    const navMenu = document.querySelector('.nav-menu');
    
    if (navToggle && navMenu) {
        navToggle.addEventListener('click', function() {
            navMenu.classList.toggle('active');
        });
    }

    // Auto-hide flash messages
    const flashMessages = document.querySelectorAll('.flash-message');
    flashMessages.forEach(function(message) {
        setTimeout(function() {
            message.style.transition = 'opacity 0.3s ease';
            message.style.opacity = '0';
            setTimeout(function() {
                message.remove();
            }, 300);
        }, 5000);
    });

    // Form validation helpers
    const forms = document.querySelectorAll('form[data-validate]');
    forms.forEach(function(form) {
        form.addEventListener('submit', function(e) {
            const requiredFields = form.querySelectorAll('[required]');
            let isValid = true;
            
            requiredFields.forEach(function(field) {
                if (!field.value.trim()) {
                    field.classList.add('error');
                    isValid = false;
                } else {
                    field.classList.remove('error');
                }
            });
            
            if (!isValid) {
                e.preventDefault();
                showError('Please fill in all required fields.');
            }
        });
    });

    // CSRF token for AJAX requests
    const csrfToken = document.querySelector('meta[name="csrf-token"]')?.getAttribute('content');
    
    function ajaxRequest(url, options = {}) {
        const headers = {
            'Content-Type': 'application/json',
            'X-CSRF-Token': csrfToken || '',
            'X-Requested-With': 'XMLHttpRequest'
        };
        
        if (options.headers) {
            Object.assign(headers, options.headers);
        }
        
        return fetch(url, {
            method: options.method || 'GET',
            headers: headers,
            body: options.body ? JSON.stringify(options.body) : null,
            credentials: 'same-origin'
        })
        .then(function(response) {
            if (!response.ok) {
                throw new Error('Network response was not ok');
            }
            return response.json();
        });
    }

    // Dashboard functionality
    if (window.location.pathname.includes('/dashboard')) {
        initDashboard();
    }

    function initDashboard() {
        // Website deletion
        const deleteButtons = document.querySelectorAll('.delete-website-btn');
        deleteButtons.forEach(function(button) {
            button.addEventListener('click', function(e) {
                e.preventDefault();
                
                if (confirm('Are you sure you want to delete this website? This action cannot be undone.')) {
                    const websiteId = this.dataset.websiteId;
                    const csrfToken = this.dataset.csrfToken;
                    
                    ajaxRequest('/dashboard/websites/' + websiteId + '/delete', {
                        method: 'POST',
                        body: { csrf_token: csrfToken }
                    })
                    .then(function(response) {
                        if (response.success) {
                            window.location.reload();
                        } else {
                            showError(response.error || 'Failed to delete website');
                        }
                    })
                    .catch(function(error) {
                        showError('An error occurred: ' + error.message);
                    });
                }
            });
        });

        // Run AI test
        const runTestButtons = document.querySelectorAll('.run-ai-test-btn');
        runTestButtons.forEach(function(button) {
            button.addEventListener('click', function(e) {
                e.preventDefault();
                
                const websiteId = this.dataset.websiteId;
                const testType = this.dataset.testType;
                const modelId = document.querySelector('#model_id')?.value;
                const csrfToken = this.dataset.csrfToken;
                
                const data = {
                    test_type: testType,
                    model_id: modelId,
                    csrf_token: csrfToken
                };
                
                if (testType === 'recommendation_test') {
                    const purpose = prompt('What is the purpose for this recommendation test?') || 'general use';
                    data.purpose = purpose;
                }
                
                this.disabled = true;
                this.textContent = 'Running...';
                
                ajaxRequest('/dashboard/websites/' + websiteId + '/run-test', {
                    method: 'POST',
                    body: data
                })
                .then(function(response) {
                    if (response.success) {
                        showSuccess('AI test queued successfully. Results will appear shortly.');
                    } else {
                        showError(response.error || 'Failed to run test');
                    }
                })
                .catch(function(error) {
                    showError('An error occurred: ' + error.message);
                })
                .finally(function() {
                    button.disabled = false;
                    button.textContent = 'Run Test';
                });
            });
        });

        // Auto-refresh job status
        const jobStatusElements = document.querySelectorAll('.job-status');
        if (jobStatusElements.length > 0) {
            setInterval(function() {
                jobStatusElements.forEach(function(element) {
                    const jobId = element.dataset.jobId;
                    if (jobId && element.textContent === 'pending' || element.textContent === 'processing') {
                        ajaxRequest('/api/job-status/' + jobId)
                            .then(function(response) {
                                if (response.status) {
                                    element.textContent = response.status;
                                    element.className = 'job-status status-' + response.status;
                                }
                            })
                            .catch(function() {
                                // Silent fail for status checks
                            });
                    }
                });
            }, 5000); // Check every 5 seconds
        }
    }

    // Admin functionality
    if (window.location.pathname.includes('/admin')) {
        initAdmin();
    }

    function initAdmin() {
        // User suspension/activation
        const toggleUserButtons = document.querySelectorAll('.toggle-user-btn');
        toggleUserButtons.forEach(function(button) {
            button.addEventListener('click', function(e) {
                e.preventDefault();
                
                const userId = this.dataset.userId;
                const action = this.dataset.action;
                const csrfToken = this.dataset.csrfToken;
                
                if (confirm('Are you sure you want to ' + action + ' this user?')) {
                    ajaxRequest('/admin/users/' + userId + '/' + action, {
                        method: 'POST',
                        body: { csrf_token: csrfToken }
                    })
                    .then(function(response) {
                        if (response.success) {
                            window.location.reload();
                        } else {
                            showError(response.error || 'Failed to ' + action + ' user');
                        }
                    })
                    .catch(function(error) {
                        showError('An error occurred: ' + error.message);
                    });
                }
            });
        });

        // Force rescan
        const rescanButtons = document.querySelectorAll('.force-rescan-btn');
        rescanButtons.forEach(function(button) {
            button.addEventListener('click', function(e) {
                e.preventDefault();
                
                const websiteId = this.dataset.websiteId;
                const csrfToken = this.dataset.csrfToken;
                
                if (confirm('This will queue new AI tests for all active models. Continue?')) {
                    this.disabled = true;
                    this.textContent = 'Processing...';
                    
                    ajaxRequest('/admin/websites/' + websiteId + '/rescan', {
                        method: 'POST',
                        body: { csrf_token: csrfToken }
                    })
                    .then(function(response) {
                        if (response.success) {
                            showSuccess('Rescan initiated successfully');
                        } else {
                            showError(response.error || 'Failed to initiate rescan');
                        }
                    })
                    .catch(function(error) {
                        showError('An error occurred: ' + error.message);
                    })
                    .finally(function() {
                        button.disabled = false;
                        button.textContent = 'Force Rescan';
                    });
                }
            });
        });
    }

    // Utility functions
    function showError(message) {
        showMessage(message, 'error');
    }

    function showSuccess(message) {
        showMessage(message, 'success');
    }

    function showMessage(message, type) {
        const existingMessage = document.querySelector('.flash-message');
        if (existingMessage) {
            existingMessage.remove();
        }

        const messageDiv = document.createElement('div');
        messageDiv.className = 'flash-message flash-' + type;
        messageDiv.textContent = message;
        document.body.appendChild(messageDiv);

        setTimeout(function() {
            messageDiv.style.transition = 'opacity 0.3s ease';
            messageDiv.style.opacity = '0';
            setTimeout(function() {
                messageDiv.remove();
            }, 300);
        }, 5000);
    }

    // Auto-resize textareas
    const textareas = document.querySelectorAll('textarea[data-autoresize]');
    textareas.forEach(function(textarea) {
        textarea.addEventListener('input', function() {
            this.style.height = 'auto';
            this.style.height = this.scrollHeight + 'px';
        });
    });

    // Password visibility toggle
    const passwordToggles = document.querySelectorAll('.password-toggle');
    passwordToggles.forEach(function(toggle) {
        toggle.addEventListener('click', function() {
            const targetId = this.dataset.target;
            const target = document.getElementById(targetId);
            
            if (target) {
                const type = target.type === 'password' ? 'text' : 'password';
                target.type = type;
                this.textContent = type === 'password' ? 'Show' : 'Hide';
            }
        });
    });

    // Smooth scrolling for anchor links
    document.querySelectorAll('a[href^="#"]').forEach(function(anchor) {
        anchor.addEventListener('click', function(e) {
            const target = document.querySelector(this.getAttribute('href'));
            if (target) {
                e.preventDefault();
                target.scrollIntoView({
                    behavior: 'smooth',
                    block: 'start'
                });
            }
        });
    });

    // Initialize tooltips (if any)
    function initTooltips() {
        const tooltips = document.querySelectorAll('[data-tooltip]');
        tooltips.forEach(function(element) {
            const tooltip = document.createElement('div');
            tooltip.className = 'tooltip';
            tooltip.textContent = element.dataset.tooltip;
            document.body.appendChild(tooltip);

            element.addEventListener('mouseenter', function() {
                const rect = element.getBoundingClientRect();
                tooltip.style.left = rect.left + rect.width / 2 + 'px';
                tooltip.style.top = rect.top - tooltip.offsetHeight - 10 + 'px';
                tooltip.style.opacity = '1';
            });

            element.addEventListener('mouseleave', function() {
                tooltip.style.opacity = '0';
            });
        });
    }

    // Initialize on DOM load
    document.addEventListener('DOMContentLoaded', function() {
        initTooltips();
    });

})();